// popup.js — full replacement (no Edge/HTTP calls)

const $ = (id) => document.getElementById(id);
const CURRENT_VERSION = chrome.runtime.getManifest().version;

// Create or return the "update required" banner (we inject it; no HTML changes needed)
function ensureBlockedBanner() {
  let banner = $('blocked-note');
  if (!banner) {
    banner = document.createElement('div');
    banner.id = 'blocked-note';
    banner.style.display = 'none';
    banner.style.background = '#181F2A';
    banner.style.color = '#fff';
    banner.style.border = '2px solid #F97316';
    banner.style.padding = '10px';
    banner.style.borderRadius = '10px';
    banner.style.margin = '10px 0';
    banner.style.fontSize = '14px';

    const h3 = document.querySelector('h3');
    if (h3 && h3.parentNode) {
      h3.parentNode.insertBefore(banner, h3.nextSibling);
    } else {
      document.body.prepend(banner);
    }
  }
  return banner;
}

function setScrapeDisabled(disabled, text = 'Update required') {
  const btn = $('scrape-search-btn');
  if (!btn) return;
  btn.disabled = !!disabled;
  if (disabled) {
    btn.style.opacity = '0.6';
    btn.style.cursor = 'not-allowed';
    btn.innerText = text;
  } else {
    btn.style.opacity = '';
    btn.style.cursor = '';
    btn.innerText = 'Get Leads from Search';
  }
}

function updateUI(connected, blockedMeta = null) {
  const dot = $('status-dot');
  const status = $('connect-status');
  const connectBtn = $('connect-btn');
  const disconnectBtn = $('disconnect-btn');

  const isBlocked = !!blockedMeta;

  // Status + buttons
  if (isBlocked) {
    if (dot) dot.classList.add('disconnected');
    if (status) status.innerText = 'Blocked (update required)';
    if (connectBtn) connectBtn.style.display = 'none';
    if (disconnectBtn) disconnectBtn.style.display = connected ? 'block' : 'none';
  } else if (connected) {
    if (dot) dot.classList.remove('disconnected');
    if (status) status.innerText = 'Connected';
    if (connectBtn) connectBtn.style.display = 'none';
    if (disconnectBtn) disconnectBtn.style.display = 'block';
  } else {
    if (dot) dot.classList.add('disconnected');
    if (status) status.innerText = 'Not Connected';
    if (connectBtn) connectBtn.style.display = 'block';
    if (disconnectBtn) disconnectBtn.style.display = 'none';
  }

  // Banner + scrape button state
  const banner = ensureBlockedBanner();
  if (isBlocked) {
    const req = blockedMeta?.required || 'latest';
    banner.textContent = `Update required. This version (v${CURRENT_VERSION}) is blocked. Please install v${req}+`;
    banner.style.display = 'block';
    setScrapeDisabled(true, 'Update required');
  } else {
    banner.style.display = 'none';
    setScrapeDisabled(!connected, connected ? '' : 'Connect first');
  }
}

// Combined render (reads once from storage)
function refresh() {
  chrome.storage.local.get(['outreachos_jwt', 'outreachos_blocked'], (result) => {
    const connected = !!result.outreachos_jwt;
    const blockedMeta = result.outreachos_blocked || null;
    updateUI(connected, blockedMeta);
  });
}

function triggerHandshake() {
  // Optional debug ping
  console.log("🔥 POPUP: Sending handshake message...");
  chrome.runtime.sendMessage({ type: "TEST_MESSAGE" }, (response) => {
    console.log("🔥 POPUP: Got response:", response);
  });

  chrome.runtime.sendMessage({ type: "OUTREACHOS_EXTENSION_HANDSHAKE" }, (response) => {
    if (response && response.success) {
      chrome.tabs.query({}, (tabs) => {
        const appTab = tabs.find(tab =>
          tab.url && (
            tab.url.startsWith("http://localhost:3000") ||
            tab.url.startsWith("https://outreachos.tech")
          )
        );
        if (appTab) {
          chrome.tabs.reload(appTab.id, {}, () => {
            chrome.tabs.update(appTab.id, { active: true }, () => {
              setTimeout(() => {
                chrome.tabs.sendMessage(appTab.id, { action: "OUTREACHOS_TRIGGER_HANDSHAKE" });
                window.close();
              }, 1000);
            });
          });
    } else {
// remove old if exists first
document.getElementById('popup-msg')?.remove();

const banner = document.createElement('div');
banner.id = 'popup-msg';
banner.style.background = '#fff';                      // high contrast white
banner.style.color = '#222';                           // readable dark text
banner.style.border = '2px solid #d9534f';             // red border
banner.style.padding = '12px 14px';
banner.style.margin = '12px 0';
banner.style.borderRadius = '8px';
banner.style.fontSize = '14px';
banner.style.lineHeight = '1.5';
banner.style.maxWidth = '320px';
banner.style.boxShadow = '0 2px 6px rgba(0,0,0,0.15)';
banner.style.display = 'flex';
banner.style.flexDirection = 'column';
banner.style.gap = '10px';

// Message text (⚠️ REMOVED — plain text only)
const msg = document.createElement('div');
msg.textContent = "OutreachOS app is not open, please open and sign in and try again.";

// Clear button
const link = document.createElement('button');
link.textContent = "Open OutreachOS App";
link.style.all = 'unset';
link.style.cursor = 'pointer';
link.style.background = '#d9534f';                     // red CTA button
link.style.color = '#fff';
link.style.fontWeight = '600';
link.style.textAlign = 'center';
link.style.padding = '10px';
link.style.borderRadius = '6px';
link.style.fontSize = '14px';
link.style.boxShadow = '0 1px 3px rgba(0,0,0,0.2)';
link.onmouseover = () => { link.style.background = '#c9302c'; };
link.onmouseout = () => { link.style.background = '#d9534f'; };

link.onclick = (e) => {
  e.preventDefault();
  chrome.tabs.query({}, (tabs) => {
    const existing = tabs.find(tab =>
      tab.url && tab.url.startsWith("https://outreachos.tech")
    );
    if (existing) {
      chrome.tabs.update(existing.id, { active: true });
    } else {
      chrome.tabs.create({ url: "https://outreachos.tech", active: true });
    }
  });
};

banner.appendChild(msg);
banner.appendChild(link);
document.body.prepend(banner);
}
      });
    } else {
      alert("Failed to open LinkedIn Messaging tab. Please try again.");
    }
  });
}

document.addEventListener("DOMContentLoaded", () => {
  refresh();

  $('connect-btn')?.addEventListener("click", () => {
    triggerHandshake();
  });

  $('disconnect-btn')?.addEventListener("click", () => {
    chrome.storage.local.remove('outreachos_jwt', () => {
      refresh();
    });
  });

  $('scrape-search-btn')?.addEventListener("click", () => {
    chrome.storage.local.get('outreachos_blocked', (res) => {
      if (res?.outreachos_blocked) {
        alert('Please update OutreachOS extension to continue.');
        return;
      }
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        const tab = tabs[0];
        if (!tab?.id) return;
        chrome.tabs.sendMessage(tab.id, { action: "OUTREACHOS_SCRAPE_SEARCH" });
        window.close();
      });
    });
  });
});

// Live update if blocked/JWT changes
chrome.storage.onChanged.addListener((changes, area) => {
  if (area !== 'local') return;
  if ('outreachos_blocked' in changes || 'outreachos_jwt' in changes) {
    refresh();
  }
});