(() => {
  if (window.__OOS_BRIDGE_INSTALLED) return;
  window.__OOS_BRIDGE_INSTALLED = true;

  const sleep = ms => new Promise(r => setTimeout(r, ms));
  const norm = s => (s || '').trim().toLowerCase().replace(/\s+/g, ' ');
  const overlays = () => [...document.querySelectorAll('.msg-overlay-conversation-bubble[role="dialog"]')];

  // Get primary header name (anchor text)
  const headerPrimary = o => norm(o.querySelector('.msg-overlay-bubble-header__title a span')?.innerText || '');

  // Get footer exact name from aria-labels
  const footerExact = o => {
    const labels = [...o.querySelectorAll('.msg-form__footer [aria-label]')].map(el => el.getAttribute('aria-label') || '');
    for (const t of labels) {
      const m = t.match(/conversation with\s+(.+)$/i);
      if (m && m[1]) return norm(m[1]);
    }
    return '';
  };

  // Check if overlay matches target name exactly
  const isExact = (o, name) => {
    const n = norm(name);
    return headerPrimary(o) === n || footerExact(o) === n;
  };

  // Find the Close button in overlay header controls
  function getCloseBtn(o) {
    const ctrl = o.querySelector('.msg-overlay-bubble-header__controls');
    if (!ctrl) return null;
    for (const b of ctrl.querySelectorAll('button')) {
      const use = b.querySelector('use');
      const href = use?.getAttribute('href') || '';
      const txt = (b.querySelector('.artdeco-button__text')?.textContent || '').trim().toLowerCase();
      if (href.includes('close') || txt.startsWith('close')) return b;
    }
    return null;
  }

  // Dispatch real mouse events to simulate user click
  function realClick(el) {
    try {
      el.dispatchEvent(new MouseEvent('mouseover', { bubbles: true }));
      el.dispatchEvent(new MouseEvent('mousedown', { bubbles: true, button: 0 }));
      el.dispatchEvent(new MouseEvent('mouseup', { bubbles: true, button: 0 }));
      el.click();
      return true;
    } catch {
      try { el.click(); return true; } catch { return false; }
    }
  }

  // Expand overlay header (un-minimize and bring to front)
  function expandOverlay(o) {
    if (!o) return;
    const header = o.querySelector('.msg-overlay-conversation-bubble-header, .msg-overlay-bubble-header');
    if (!header) return;
    o.scrollIntoView({ block: 'center' });
    if (o.getBoundingClientRect().height < 150) header.click();
    header.click();
  }

  // Close all overlays except the target overlay by clicking their Close buttons
  async function closeOtherOverlaysExact(name, rounds = 5) {
    const n = norm(name);
    for (let i = 0; i < rounds; i++) {
      const ovs = overlays();
      let closedCount = 0;
      for (const o of ovs) {
        if (isExact(o, name)) continue;
        expandOverlay(o);
        await sleep(50);
        const btn = getCloseBtn(o);
        if (btn) {
          realClick(btn);
          closedCount++;
          await sleep(150);
        }
      }
      if (closedCount === 0) break;
      await sleep(400);
    }
  }

  // Get editor element inside overlay
  function getEditor(o) {
    return o.querySelector('div.msg-form__contenteditable[contenteditable="true"]') ||
           o.querySelector('div[contenteditable="true"][role="textbox"]') || null;
  }

  // Get Send button inside overlay
  function getSend(o) {
    return o.querySelector('button.msg-form__send-button[type="submit"]') ||
           o.querySelector('button.msg-form__send-button') ||
           o.querySelector('button[aria-label^="Send"]') || null;
  }

  // Set caret to end of editor content
  function setCaretToEnd(editor) {
    const sel = window.getSelection();
    const range = document.createRange();
    if (!editor.firstChild) editor.appendChild(document.createElement('p'));
    const last = editor.lastChild;
    if (last.nodeType === Node.TEXT_NODE) {
      range.setStart(last, last.textContent.length);
    } else {
      range.selectNodeContents(editor);
      range.collapse(false);
    }
    sel.removeAllRanges();
    sel.addRange(range);
  }

  // Type message with real keyboard and input events
  async function typeWithEvents(editor, text) {
    editor.click();
    editor.focus();
    setCaretToEnd(editor);

    try {
      editor.dispatchEvent(new InputEvent('beforeinput', { bubbles: true, inputType: 'deleteContentBackward' }));
    } catch {}
    try { document.execCommand('selectAll', false, null); document.execCommand('delete', false, null); } catch {}

    let bulkOk = true;
    try {
      const ok = document.execCommand('insertText', false, text);
      if (!ok) bulkOk = false;
    } catch { bulkOk = false; }

    if (!bulkOk || (editor.innerText || editor.textContent || '').trim() !== text.trim()) {
      editor.innerHTML = '';
      for (const ch of text) {
        editor.dispatchEvent(new KeyboardEvent('keydown', { key: ch, bubbles: true }));
        try { document.execCommand('insertText', false, ch); } catch {
          editor.appendChild(document.createTextNode(ch));
        }
        editor.dispatchEvent(new InputEvent('input', { bubbles: true, data: ch, inputType: 'insertText' }));
        editor.dispatchEvent(new KeyboardEvent('keyup', { key: ch, bubbles: true }));
        await sleep(10);
      }
    } else {
      editor.dispatchEvent(new InputEvent('input', { bubbles: true, data: text, inputType: 'insertText' }));
    }

    editor.dispatchEvent(new Event('change', { bubbles: true }));
    editor.dispatchEvent(new KeyboardEvent('keyup', { key: 'End', bubbles: true }));
    await sleep(60);

    const ov = editor.closest('.msg-overlay-conversation-bubble');
    const sendBtn = ov && getSend(ov);
    if (sendBtn && (sendBtn.disabled || sendBtn.getAttribute('aria-disabled') === 'true')) {
      try { document.execCommand('insertText', false, ' '); } catch { editor.appendChild(document.createTextNode(' ')); }
      editor.dispatchEvent(new InputEvent('input', { bubbles: true, data: ' ', inputType: 'insertText' }));
      await sleep(40);
      try { document.execCommand('delete', false, null); } catch {
        const t = editor.innerText || editor.textContent || '';
        editor.innerHTML = '';
        editor.appendChild(document.createTextNode(t.slice(0, -1)));
      }
      editor.dispatchEvent(new InputEvent('input', { bubbles: true, inputType: 'deleteContentBackward' }));
      await sleep(60);
    }
  }

  // Listen for messages from content script
  window.addEventListener('message', async (event) => {
    const data = event.data || {};
    if (data.type !== 'OOS_BRIDGE_SEND_DM') return;
    const reqId = data.reqId;
    try {
      // Wait for overlay to appear
      let overlay = null;
      for (let i = 0; i < 30; i++) {
        overlay = overlays().find(o => isExact(o, data.name));
        if (overlay) break;
        await sleep(200);
      }
      if (!overlay) throw new Error('Overlay not found');

      // Close other overlays
      await closeOtherOverlaysExact(data.name);

      // Re-find overlay after close
      overlay = overlays().find(o => isExact(o, data.name));
      if (!overlay) throw new Error('Overlay disappeared after closing others');

      // Type message
      const editor = getEditor(overlay);
      if (!editor) throw new Error('Editor not found');
      await typeWithEvents(editor, data.message);

      // Click send
      const sendBtn = getSend(overlay);
      if (!sendBtn) throw new Error('Send button not found');
      sendBtn.click();

      window.postMessage({ type: 'OOS_BRIDGE_SEND_DM_RESULT', reqId, ok: true }, '*');
    } catch (error) {
      window.postMessage({ type: 'OOS_BRIDGE_SEND_DM_RESULT', reqId, ok: false, error: error.message }, '*');
    }
  });

  // Signal ready
  window.postMessage({ type: 'OOS_BRIDGE_READY' }, '*');
})();