let scraping = false;

const pause = (ms) => new Promise(res => setTimeout(res, ms));

// ===================================================================================================
// 🔥 FIX: Environment-aware URL helper to consolidate API endpoints and reduce duplicate calls
// This prevents mixed localhost/production URL usage that was causing duplicate function invocations
// ===================================================================================================
const getApiUrl = (endpoint) => {
  // Auto-detect environment based on current context
  let baseUrl;
  try {
    // If we're in a web context, check the current location
    baseUrl = (typeof window !== 'undefined' && window.location?.hostname === 'localhost')
      ? 'http://localhost:3000/api'
      : 'https://outreachos.tech/api';
  } catch (e) {
    // Fallback to production if detection fails
    baseUrl = 'https://outreachos.tech/api';
  }
  return `${baseUrl}/${endpoint}`;S
};

// 🔥 UPDATED: Check both pause flags and state machine coordination
async function isPaused() {
  return new Promise(res => {
    chrome.storage.local.get(['outreachos_sync_paused', 'messageBeingSent'], result => {
      const paused = !!result.outreachos_sync_paused || !!result.messageBeingSent;
      res(paused);
    });
  });
}

chrome.storage.local.get(['outreachos_main_tab_id'], (stored) => {
  chrome.runtime.sendMessage({ action: 'OUTREACHOS_GET_TAB_ID' }, (resp) => {
    if (resp && stored.outreachos_main_tab_id === resp.tabId) {
      waitForScrollAndSupabase();
    } else {
      console.log('[OutreachOS] Not main tab – sync disabled');
    }
  });
});

// ===================================================================================================
// 🔥 FIX: Removed duplicate legacy initialization code that was causing excessive API calls
// This function was calling get-sync-status and update-sync-progress redundantly with background.js
// Now we only prepare the UI and let the state machine in background.js handle all sync logic
// ===================================================================================================
function waitForScrollAndSupabase() {
  waitForElement('.msg-conversations-container__conversations-list', async () => {
    const loaded = await scrollSidebarToLoadAll(50);
    console.log('[OutreachOS] Loaded conversations - letting state machine handle sync initialization');
    
    // ✅ FEATURE PRESERVED: Conversation loading still works exactly the same
    // ✅ FEATURE PRESERVED: JWT waiting logic moved to background.js state machine
    // ✅ API OPTIMIZATION: Removed duplicate get-sync-status call (background.js handles this)
    // ✅ API OPTIMIZATION: Removed duplicate first sync initialization (background.js handles this)
    
    console.log('[OutreachOS] messaging-sync.js ready - awaiting state machine commands');
  });
}

// ===================================================================================================
// 🔥 FIX: Removed startAutoScraping function - this was causing duplicate update-sync-progress calls
// The background.js state machine now handles ALL initial scraping via OUTREACHOS_FULL_SCRAPE message
// This eliminates the duplicate code path that was causing high Vercel function usage
// ===================================================================================================
// ✅ FEATURE PRESERVED: Full scraping functionality moved to state machine-triggered path (line 524+)
// ✅ FEATURE PRESERVED: All scraping capabilities maintained via message handlers
// ✅ API OPTIMIZATION: Eliminates duplicate update-sync-progress calls during first sync

function resolveDateLabel(label) {
  const lower = label.trim().toLowerCase();
  const now = new Date();
  if (lower === 'today') return new Date(now.toDateString());
  if (lower === 'yesterday') { 
    const y = new Date(); 
    y.setDate(now.getDate() - 1); 
    return new Date(y.toDateString()); 
  }
  const weekday = ['sunday','monday','tuesday','wednesday','thursday','friday','saturday'].indexOf(lower);
  if (weekday !== -1) { 
    const diff = (now.getDay() - weekday + 7) % 7; 
    const d = new Date(); 
    d.setDate(now.getDate() - diff); 
    return new Date(d.toDateString()); 
  }
  const parsed = new Date(`${label}, ${now.getFullYear()}`);
  return isNaN(parsed) ? null : parsed;
}

function hashString(str){
  let h = 0, i, c; 
  if(!str) return h;
  for(i = 0; i < str.length; i++) {
    c = str.charCodeAt(i);
    h = ((h << 5) - h) + c;
    h |= 0;
  }
  return Math.abs(h).toString();
}

function scrapeMessages(existingOrdinals = {}) {
  const msgs = [];
  let dateCtx = null;
  let tsIso   = '';

  document.querySelectorAll('.msg-s-message-list__event').forEach(ev => {
    const dateEl = ev.querySelector('.msg-s-message-list__time-heading');
    if (dateEl) dateCtx = resolveDateLabel(dateEl.textContent.trim());

    const timeEl = ev.querySelector('.msg-s-message-group__timestamp');
    if (dateCtx && timeEl) {
      const combo = new Date(`${dateCtx.toDateString()} ${timeEl.textContent.trim()}`);
      if (!isNaN(combo)) tsIso = combo.toISOString();
    }

    ev.querySelectorAll('.msg-s-event-listitem').forEach(div => {
      const fromMe  = !div.classList.contains('msg-s-event-listitem--other');
      const bodyEl  = div.querySelector('.msg-s-event-listitem__body');
      const content = (bodyEl?.textContent || '').trim();

      const lc = content.toLowerCase();
      if ((lc.includes('view') && lc.includes('profile')) || lc.includes('connection request')) return;
      if (!content) return;

      const audio = div.querySelector('button.msg-s-event-listitem__audio') ? 'true' : 'false';

      let image = null;
      const imgEl = div.querySelector('.msg-s-event-listitem__image');
      if (imgEl) {
        const raw = imgEl.src || imgEl.getAttribute('data-delayed-url') || '';
        const clean = raw.match(/https:\/\/[^\s"<>]+/i);
        if (clean) image = clean[0];
      }

      const linkMatch = content.match(/https?:\/\/[^\s]+/);
      const link = linkMatch ? linkMatch[0] : null;

      const uidData = [
        getConversationProfileUrl(),
        tsIso,
        content,
        image || '',
        audio,
        link || ''
      ].join('_');
      const uid = hashString(uidData);
      const ordinal = existingOrdinals[uid] ?? msgs.length;

      msgs.push({
        id: uid,
        from_me: fromMe,
        content,
        timestamp: tsIso,
        ordinal,
        image: image || null,
        audio: audio,
        link: link || null
      });
    });
  });

  return msgs;
}

function getConversationProfileUrl(){
  return document.querySelector('.msg-thread__link-to-profile')?.href || '';
}

function getConversationName(){
  return (
    document.querySelector('.msg-thread__link-to-profile span[aria-hidden="true"]')?.textContent.trim() ||
    document.querySelector('.msg-thread__conversation-title')?.textContent.trim() ||
    document.querySelector('.msg-conversation-header__participants')?.textContent.trim() ||
    'Unknown'
  );
}

// ===================================================================================================
// 🔥 FIX: Updated to use environment-aware URL helper instead of hardcoded localhost URL
// This ensures consistent API endpoint usage across different environments
// ===================================================================================================
function sendSyncWithToken(linkedin_url, name, messages, attempt = 0) {
  chrome.storage.local.get('outreachos_jwt', ({ outreachos_jwt }) => {
    if (!outreachos_jwt) {
      if (attempt < 10) {
        setTimeout(() => sendSyncWithToken(linkedin_url, name, messages, attempt + 1), 1000);
      } else {
        console.error('[DEBUG] Gave up waiting for JWT after 10 seconds.');
      }
      return;
    }
    
    // ✅ API OPTIMIZATION: Using environment-aware URL helper
    fetch(getApiUrl('sync-messages'), {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        Authorization: `Bearer ${outreachos_jwt}`
      },
      body: JSON.stringify({
        linkedin_url,
        name,
        messages,
      })
    })
    .then(r => r.text())
    .then(text => {
      try {
        const json = JSON.parse(text);
        console.log('[DEBUG] Parsed JSON:', json);
      } catch (e) {
        console.error('[DEBUG] Failed to parse JSON. Likely received HTML. Error:', e);
      }
    })
    .catch(err => console.error('[DEBUG] API fetch error:', err));
  });
}

function syncMessages(linkedin_url, name, messages) {
  sendSyncWithToken(linkedin_url, name, messages);
}

// ===================================================================================================
// 🔥 FIX: Updated outbox functions to use environment-aware URL helper
// These functions handle message verification and status updates - no feature changes
// ===================================================================================================
// 🔥 NEW: Get failed outbox messages for specific participant
async function getFailedOutboxMessages(participantName) {
  return new Promise((resolve) => {
    chrome.storage.local.get('outreachos_jwt', async ({ outreachos_jwt }) => {
      if (!outreachos_jwt) {
        resolve([]);
        return;
      }

      try {
        // Get failed messages from last 30 minutes for this participant
        const thirtyMinutesAgo = new Date(Date.now() - 30 * 60 * 1000).toISOString();
        
        // ✅ API OPTIMIZATION: Using environment-aware URL helper
        const response = await fetch(`${getApiUrl('get-failed-outbox')}?participant_name=${encodeURIComponent(participantName)}&since=${encodeURIComponent(thirtyMinutesAgo)}`, {
          headers: { Authorization: `Bearer ${outreachos_jwt}` }
        });

        if (response.ok) {
          const data = await response.json();
          resolve(data || []);
        } else {
          resolve([]);
        }
      } catch (error) {
        console.error('[OUTBOX] Error fetching failed messages:', error);
        resolve([]);
      }
    });
  });
}

// 🔥 NEW: Update outbox message status
async function updateOutboxStatus(messageId, status) {
  return new Promise((resolve) => {
    chrome.storage.local.get('outreachos_jwt', async ({ outreachos_jwt }) => {
      if (!outreachos_jwt) {
        resolve(false);
        return;
      }

      try {
        // ✅ API OPTIMIZATION: Using environment-aware URL helper
        const response = await fetch(getApiUrl('update-outbox-message'), {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            Authorization: `Bearer ${outreachos_jwt}`
          },
          body: JSON.stringify({
            id: messageId,
            status: status,
            sent_at: status === 'sent' ? new Date().toISOString() : null
          })
        });

        const success = response.ok;
        if (success) {
          console.log(`[OUTBOX] Updated message ${messageId} to ${status}`);
        }
        resolve(success);
      } catch (error) {
        console.error('[OUTBOX] Error updating message status:', error);
        resolve(false);
      }
    });
  });
}

// 🔥 NEW: Cross-reference scraped messages with failed outbox messages
async function checkAndUpdateFailedMessages(scrapedMessages, participantName) {
  try {
    const failedMessages = await getFailedOutboxMessages(participantName);
    if (failedMessages.length === 0) return;

    console.log(`[OUTBOX] Found ${failedMessages.length} failed messages for ${participantName}`);

    for (const failedMsg of failedMessages) {
      // Look for matching scraped message (from_me = true and similar content)
      const match = scrapedMessages.find(scraped => 
        scraped.from_me && 
        scraped.content && 
        failedMsg.message &&
        (
          // Exact match
          scraped.content.trim() === failedMsg.message.trim() ||
          // Partial match (first 100 chars)
          scraped.content.substring(0, 100) === failedMsg.message.substring(0, 100) ||
          // Reverse partial match
          failedMsg.message.substring(0, 100) === scraped.content.substring(0, 100)
        )
      );

      if (match) {
        console.log(`[OUTBOX] Found match for failed message ${failedMsg.id}`);
        await updateOutboxStatus(failedMsg.id, 'sent');
      }
    }
  } catch (error) {
    console.error('[OUTBOX] Error checking failed messages:', error);
  }
}
function waitForElement(sel, cb){
  const el = document.querySelector(sel);
  if (el) return cb(el);
  const obs = new MutationObserver(() => {
    const node = document.querySelector(sel);
    if (node) { 
      obs.disconnect(); 
      cb(node); 
    }
  });
  obs.observe(document.body, {childList: true, subtree: true});
}

async function scrollSidebarToLoadAll(maxCount = 50) {
  const sidebar = document.querySelector('.msg-conversations-container__conversations-list');
  if (!sidebar) return 0;

  let lastCount = 0;
  let stableCount = 0;
  for (let i = 0; i < 40; i++) {
    sidebar.scrollTop = sidebar.scrollHeight;
    sidebar.dispatchEvent(new Event('scroll', { bubbles: true }));
    await pause(1000);

    const items = document.querySelectorAll('li[class*="msg-conversation-listitem"]');
    const currentCount = items.length;

    if (currentCount >= maxCount) break;
    if (currentCount === lastCount) {
      stableCount++;
      if (stableCount >= 5) break;
    } else {
      stableCount = 0;
    }
    lastCount = currentCount;
  }

  const finalCount = document.querySelectorAll('li[class*="msg-conversation-listitem"]').length;
  console.log('[OutreachOS] Loaded conversations:', finalCount);
  return finalCount;
}

function fetchExistingOrdinals(linkedin_url){
  return new Promise(res => {
    chrome.runtime.sendMessage(
      { action:'OUTREACHOS_GET_ORDINALS', linkedin_url },
      resp => res(resp?.data || {})
    );
  });
}

// 🔥 UPDATED: Simplified scraping with state machine coordination + outbox status updates
async function syncAllConversations(fullScrape = false, maxConvos = 50, onDone) {
  if (scraping) {
    console.log('[OutreachOS] Scraping already in progress, skipping...');
    return;
  }
  
  // 🔥 UPDATED: Check if paused (includes messageBeingSent check)
  if (await isPaused()) {
    console.log('[OutreachOS] Scraping is paused, skipping...');
    return;
  }
  
  scraping = true;
  console.log(`[OutreachOS] Starting ${fullScrape ? 'FULL' : 'INCREMENTAL'} scrape...`);

  const state = await chrome.storage.local.get(['outreachos_full_scrape_index', 'outreachos_full_scrape_done']);
  
  const links = Array.from(document.querySelectorAll('li[class*="msg-conversation-listitem"] .msg-conversation-listitem__link'));
  let idx = fullScrape ? (state.outreachos_full_scrape_index || 0) : 0;
  const hardCap = fullScrape ? links.length : Math.min(links.length, maxConvos);

  console.log('[Sync] Starting at idx:', idx, 'HardCap:', hardCap);

  return new Promise((resolve) => {
    const cycle = async () => {
      // 🔥 UPDATED: Check pause status in each cycle
      if (await isPaused()) {
        console.log('[OutreachOS] Scraping paused during cycle, waiting...');
        setTimeout(cycle, 1000);
        return;
      }

      if (idx >= hardCap) {
        console.log('[OutreachOS] ✅ Sync loop finished');
        // 🔥 UPDATED: Send completion message to state machine
        chrome.runtime.sendMessage({ action: 'MESSAGING_SCRAPE_DONE' });
        scraping = false;
        chrome.runtime.sendMessage({action:'OUTREACHOS_SYNC_PROGRESS', payload:{done:idx,total:hardCap}});
        if (fullScrape && onDone) onDone(idx);
        resolve(idx);
        return;
      }

      const link = links[idx];
      if (!link) {
        console.warn(`[OutreachOS] No link found at index ${idx}, skipping...`);
        idx++;
        setTimeout(cycle, 100);
        return;
      }

      const li = link.closest('li[class*="msg-conversation-listitem"]');
      const name = li?.querySelector('.msg-conversation-listitem__participant-names .truncate')?.textContent.trim() || 'Unknown';

      console.log(`[OutreachOS] Processing conversation ${idx + 1}/${hardCap}: ${name}`);

      try {
        link.click();
      } catch (e) {
        console.warn('Click failed:', e);
      }

      // 🔥 UPDATED: Slightly longer delay for stability
      const randomDelay = Math.random() * 500 + 500; // 500-1000ms
      await pause(randomDelay);

      let msgs = [];
      const linkedin_url = getConversationProfileUrl();

      if (fullScrape) {
        await scrollMessageListToTop();
        msgs = scrapeMessages();
      } else {
        const ord = await fetchExistingOrdinals(linkedin_url);
        msgs = scrapeMessages(ord);
      }

      console.log(`[SCRAPE] idx: ${idx}, progress: ${idx + 1}, hardCap: ${hardCap}, links.length: ${links.length}`);
      console.log(`[${idx + 1}/${hardCap}] ${name} → ${msgs.length} msgs`);

      if (linkedin_url) {
        // 🔥 SYNC MESSAGES TO DATABASE
        syncMessages(linkedin_url, name, msgs);
        
        // 🔥 NEW: Check and update failed outbox messages
        await checkAndUpdateFailedMessages(msgs, name);
      } else {
        console.warn('⚠️  No linkedin_url, skipped');
      }

      idx++;
      if (fullScrape) chrome.storage.local.set({ outreachos_full_scrape_index: idx });

      // 🔥 UPDATED: Small delay before next iteration
      setTimeout(cycle, 200);
    };

    cycle();
  });
}

async function scrollMessageListToTop() {
  const container = document.querySelector('.msg-s-message-list__events');
  if (!container) return;
  container.scrollTop = 0;
  await pause(300);
}

// 🔥 UPDATED: Message handlers for state machine coordination
chrome.runtime.onMessage.addListener((req, sender, sendResponse) => {
  // 🔥 UPDATED: Handle incremental scrape requests from state machine
  if (req.action === 'OUTREACHOS_INCREMENTAL_SCRAPE') {
    console.log('[OutreachOS] Received incremental scrape request from state machine');
    
    (async () => {
      try {
        if (scraping) {
          console.log('[OutreachOS] Already scraping, ignoring request');
          sendResponse({ ok: false, error: 'Already scraping' });
          return;
        }

        if (await isPaused()) {
          console.log('[OutreachOS] Scraping is paused, ignoring request');
          sendResponse({ ok: false, error: 'Scraping paused' });
          return;
        }

        console.log('[OutreachOS] Starting incremental scrape...');
        await scrollSidebarToLoadAll(20);
        const result = await syncAllConversations(false, 20);
        console.log('[OutreachOS] Incremental scrape completed:', result);
        sendResponse({ ok: true, processed: result });
      } catch (error) {
        console.error('[OutreachOS] Error in incremental scrape:', error);
        sendResponse({ ok: false, error: error.message });
      }
    })();
    
    return true; // Keep message channel open for async response
  }

  // 🔥 NEW: Handle full scrape requests from state machine
  if (req.action === 'OUTREACHOS_FULL_SCRAPE') {
    console.log('[OutreachOS] Received full scrape request from state machine');
    
    (async () => {
      try {
        if (scraping) {
          console.log('[OutreachOS] Already scraping, ignoring request');
          sendResponse({ ok: false, error: 'Already scraping' });
          return;
        }

        console.log('[OutreachOS] Starting full scrape...');
        const loaded = await scrollSidebarToLoadAll(50);
        
        let scrapedCount = 0;
        await syncAllConversations(true, loaded, (progress) => {
          scrapedCount = progress;
        });

        // ===================================================================================================
        // 🔥 FIX: Added guard to prevent duplicate update-sync-progress calls
        // This was the main cause of excessive API usage - now only updates on first completion
        // ===================================================================================================
        chrome.storage.local.get(['outreachos_jwt', 'outreachos_full_scrape_done'], async ({ outreachos_jwt, outreachos_full_scrape_done }) => {
          if (outreachos_jwt && !outreachos_full_scrape_done) {
            // ✅ API OPTIMIZATION: Only update sync progress if this is the first completion
            console.log('[SYNC PROGRESS] First sync completion - updating progress');
            try {
              // ✅ API OPTIMIZATION: Using environment-aware URL helper
              const resp = await fetch(getApiUrl('update-sync-progress'), {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                  Authorization: `Bearer ${outreachos_jwt}`
                },
                body: JSON.stringify({ sync_progress: scrapedCount, first_sync_done: true })
              });
              const json = await resp.json();
              console.log('[SYNC PROGRESS] First sync update response:', json);
              
              // ✅ FEATURE PRESERVED: Mark completion to prevent future duplicate calls
              chrome.storage.local.set({ outreachos_full_scrape_done: true });
            } catch (error) {
              console.error('[SYNC PROGRESS] Failed to update:', error);
            }
          } else if (outreachos_full_scrape_done) {
            // ✅ API OPTIMIZATION: Skip duplicate calls on subsequent full scrapes
            console.log('[SYNC PROGRESS] Skipping duplicate update-sync-progress call (already completed)');
          }
        });

        console.log('[OutreachOS] Full scrape completed:', scrapedCount);
        sendResponse({ ok: true, processed: scrapedCount });
      } catch (error) {
        console.error('[OutreachOS] Error in full scrape:', error);
        sendResponse({ ok: false, error: error.message });
      }
    })();
    
    return true; // Keep message channel open for async response
  }

  // 🔥 LEGACY: Keep existing sync progress handler for compatibility
  if (req.action === 'OUTREACHOS_SYNC_PROGRESS') {
    console.log('[OutreachOS] Sync progress update:', req.payload);
    sendResponse({ ok: true });
    return true;
  }
});

// 🔥 REMOVED: Old interval-based scraping system
// The state machine in background.js now controls all scraping operations

console.log('[OutreachOS] messaging_sync.js loaded - waiting for state machine commands');