// --- Helper: Wait for an element to appear in the DOM ---
function waitForElement(selector, callback) {
  const el = document.querySelector(selector);
  if (el) {
    callback(el);
    return;
  }
  const observer = new MutationObserver(() => {
    const target = document.querySelector(selector);
    if (target) {
      observer.disconnect();
      callback(target);
    }
  });
  observer.observe(document.body, { childList: true, subtree: true });
}

// --- Date label normalization ---
function resolveDateLabel(label) {
  const lower = label.trim().toLowerCase();
  const now = new Date();
  if (lower === 'today') return new Date(now.toDateString());
  if (lower === 'yesterday') {
    const yesterday = new Date();
    yesterday.setDate(now.getDate() - 1);
    return new Date(yesterday.toDateString());
  }
  const days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
  const weekdayIndex = days.indexOf(lower);
  if (weekdayIndex !== -1) {
    const nowDay = now.getDay();
    const diff = (nowDay - weekdayIndex + 7) % 7;
    const date = new Date();
    date.setDate(now.getDate() - diff);
    return new Date(date.toDateString());
  }
  const parsed = new Date(`${label}, ${now.getFullYear()}`);
  if (!isNaN(parsed.getTime())) return parsed;
  return null;
}

// --- Hash function for unique message id ---
function hashString(str) {
  let hash = 0, i, chr;
  if (str.length === 0) return hash;
  for (i = 0; i < str.length; i++) {
    chr   = str.charCodeAt(i);
    hash  = ((hash << 5) - hash) + chr;
    hash |= 0;
  }
  return Math.abs(hash).toString();
}

// --- LinkedIn profile URL ---
function getConversationProfileUrl() {
  const profileLink = document.querySelector('.msg-thread__link-to-profile');
  return profileLink ? profileLink.href : '';
}

// --- Conversation name ---
function getConversationName() {
  const nameEl = document.querySelector('.msg-thread__link-to-profile span[aria-hidden="true"]');
  if (nameEl) return nameEl.textContent.trim();
  const headerEl = document.querySelector('.msg-thread__conversation-title');
  if (headerEl) return headerEl.textContent.trim();
  const participantsEl = document.querySelector('.msg-conversation-header__participants');
  if (participantsEl) return participantsEl.textContent.trim();
  return 'Unknown';
}

// --- Robust message scraper: gets all messages, correct order, correct timestamp ---
function scrapeMessages() {
  const messages = [];
  let currentDateContext = null;
  let currentTimestamp = '';
  let ordinalCounter = 0;
  const allNodes = document.querySelectorAll('.msg-s-message-list__event');
  allNodes.forEach((msgEl, idx) => {
    // Update day label if present
    const dateEl = msgEl.querySelector('.msg-s-message-list__time-heading');
    if (dateEl) {
      const label = dateEl.textContent.trim();
      const resolved = resolveDateLabel(label);
      if (resolved) {
        currentDateContext = resolved;
      }
    }
    // Update timestamp if present
    const timeEl = msgEl.querySelector('.msg-s-message-group__timestamp');
    const timeStr = timeEl ? timeEl.textContent.trim() : '';
    if (currentDateContext && timeStr) {
      const combined = new Date(`${currentDateContext.toDateString()} ${timeStr}`);
      if (!isNaN(combined)) {
        currentTimestamp = combined.toISOString();
      }
    }
    // For every message in this event
    const eventDivs = msgEl.querySelectorAll('.msg-s-event-listitem');
    eventDivs.forEach((eventDiv) => {
      const fromMe = !eventDiv.classList.contains('msg-s-event-listitem--other');
      const contentEl = eventDiv.querySelector('.msg-s-event-listitem__body');
      let content = contentEl ? contentEl.textContent.trim() : '';
      // Skip system/profile messages
      if (
        content.toLowerCase().includes("view") && content.toLowerCase().includes("profile") ||
        content.toLowerCase().includes("connection request")
      ) return;
      // Unique id: hash of profile + timestamp + content + ordinalCounter
      const uniqueString = `${getConversationProfileUrl()}_${currentTimestamp}_${content}_${ordinalCounter}`;
      messages.push({
        id: hashString(uniqueString),
        from_me: fromMe,
        content,
        timestamp: currentTimestamp,
        ordinal: ordinalCounter++
      });
    });
  });
  return messages;
}

// --- Robust scroll up in the message list to load full chat history (simulates real user) ---
async function scrollMessageListToTop(progressCallback) {
  const msgList = document.querySelector('.msg-s-message-list-content');
  if (!msgList) return;
  let lastCount = 0;
  let stableCount = 0;
  for (let i = 0; i < 50; i++) {
    for (let j = 0; j < 10; j++) {
      msgList.scrollBy(0, -100);
      msgList.dispatchEvent(new Event('scroll'));
      await new Promise(res => setTimeout(res, 100));
    }
    let spinner = document.querySelector('.artdeco-loader:not(.hidden)');
    let spinnerWait = 0;
    while (spinner && spinnerWait < 10) {
      await new Promise(res => setTimeout(res, 500));
      spinner = document.querySelector('.artdeco-loader:not(.hidden)');
      spinnerWait++;
    }
    await new Promise(res => setTimeout(res, 1200));
    const currentCount = document.querySelectorAll('.msg-s-message-list__event').length;
    if (progressCallback) progressCallback(currentCount);
    if (currentCount === lastCount) {
      stableCount++;
      if (stableCount >= 3) break;
    } else {
      stableCount = 0;
    }
    lastCount = currentCount;
  }
}

// --- Send messages to backend API ---
function syncMessages(linkedin_url, name, messages) {
  if (!linkedin_url || messages.length === 0) return;
  chrome.runtime.sendMessage({
    action: "OUTREACHOS_SYNC_MESSAGES",
    payload: {
      linkedin_url,
      name,
      messages,
    }
  });
}

// --- Inject "Full Sync" button beside each conversation in the sidebar ---
function injectFullSyncButtons() {
  document.querySelectorAll('li[class*="msg-conversation-listitem"]').forEach(li => {
    if (li.querySelector('.outreachos-fullsync-btn')) return; // Avoid duplicates
    const btn = document.createElement('button');
    btn.textContent = "Full Sync";
    btn.className = "outreachos-fullsync-btn";
    btn.style.marginLeft = "8px";
    btn.style.background = "#7C3AED";
    btn.style.color = "#fff";
    btn.style.border = "none";
    btn.style.borderRadius = "4px";
    btn.style.padding = "2px 8px";
    btn.style.cursor = "pointer";
    btn.onclick = (e) => {
      e.stopPropagation();
      li.click(); // Focus this conversation

      // Show a non-blocking banner at the top of the chatbox
      let banner = document.getElementById('outreachos-fullsync-banner');
      if (!banner) {
        banner = document.createElement('div');
        banner.id = 'outreachos-fullsync-banner';
        banner.style.position = 'fixed';
        banner.style.top = '70px';
        banner.style.left = '50%';
        banner.style.transform = 'translateX(-50%)';
        banner.style.background = '#7C3AED';
        banner.style.color = '#fff';
        banner.style.padding = '10px 24px';
        banner.style.borderRadius = '8px';
        banner.style.zIndex = '999999';
        banner.style.fontWeight = 'bold';
        banner.innerHTML = `
          Please scroll up in the chatbox until you reach the very first message.<br>
          <button id="outreachos-fullsync-start-btn" style="margin-top:8px;background:#22c55e;color:#fff;padding:6px 18px;border:none;border-radius:6px;font-size:14px;cursor:pointer;">Start Sync</button>
          <div id="outreachos-fullsync-progress" style="margin-top:8px;height:6px;background:#23283a;border-radius:4px;overflow:hidden;width:100%;display:none;">
            <div id="outreachos-fullsync-progress-bar" style="background:#22c55e;height:6px;width:0%;transition:width 0.3s;"></div>
          </div>
          <div id="outreachos-fullsync-status" style="margin-top:8px;font-size:13px;"></div>
        `;
        document.body.appendChild(banner);
      }
      document.getElementById('outreachos-fullsync-start-btn').onclick = async () => {
        // Show progress bar
        const progressDiv = document.getElementById('outreachos-fullsync-progress');
        const progressBar = document.getElementById('outreachos-fullsync-progress-bar');
        const statusDiv = document.getElementById('outreachos-fullsync-status');
        progressDiv.style.display = 'block';
        statusDiv.textContent = "Sync started! Please wait...";
        let lastCount = 0;
        await scrollMessageListToTop((count) => {
          const pct = Math.min(100, Math.round((count / 200) * 100));
          progressBar.style.width = `${pct}%`;
          lastCount = count;
        });
        progressBar.style.width = '100%';
        setTimeout(() => {
          progressDiv.style.display = 'none';
          progressBar.style.width = '0%';
        }, 1000);
        const messages = scrapeMessages();
        const linkedin_url = getConversationProfileUrl();
        const name = getConversationName();
        chrome.runtime.sendMessage({
          action: "OUTREACHOS_SYNC_MESSAGES",
          payload: { linkedin_url, name, messages }
        });
        statusDiv.textContent = "✅ Sync started! Please wait for the sync to complete.";
        setTimeout(() => banner.remove(), 3000);
      };
    };
    // Insert the button after the participant name
    const nameDiv = li.querySelector('.msg-conversation-listitem__participant-names');
    if (nameDiv) nameDiv.appendChild(btn);
  });
}

// --- Inject "Full Sync" button on sidebar load and whenever the sidebar changes ---
waitForElement('.msg-conversations-container__conversations-list', () => {
  injectFullSyncButtons();
  const sidebar = document.querySelector('.msg-conversations-container__conversations-list');
  const observer = new MutationObserver(() => injectFullSyncButtons());
  observer.observe(sidebar, { childList: true, subtree: true });
});