console.log("OutreachOS content.js loaded on", window.location.href);
console.log("🔥 CONNECT.JS LOADED:", window.location.href);

/* ==========  WORKFLOW-DM LOGGER  ========== */
const DM_LOG = (() => {
  const pad = (n, c = ' ') => String(n).padEnd(4, c);
  const fmt = () => new Date().toISOString().substr(11, 12);
  let depth = 0;
  const api = (msg, obj) => {
    depth++;
    console.log(`[${fmt()}] ${pad(depth - 1, '│  ')}${msg}` + (obj ? ' ' + JSON.stringify(obj).slice(0, 200) : ''));
  };
  api.exit = () => depth = Math.max(0, depth - 1);
  return api;
})();

let ENV = null;
const getEnv = async () => {
  if (ENV) return ENV;

  let envUrl;
  if (location.hostname === 'localhost' && location.port === '3000') {
    envUrl = 'http://localhost:3000/api/env';
  } else {
    envUrl = 'https://outreachos.tech/api/env'; // ✅ use frontend, not api.*
  }

  const res = await fetch(envUrl);
  ENV = await res.json();
  return ENV;
};

// ============================================================================
// Auto JWT bridge from web app (localhost/outreachos.tech) → extension storage
// Keeps JWT in extension up to date without manual copy/paste
// ============================================================================
(function autoSetJWTFromWebApp() {
  const isWebAppHost =
    (location.hostname === 'localhost' && location.port === '3000') ||
    location.hostname.endsWith('outreachos.tech');

  if (!isWebAppHost) return;

  // Wait for Supabase to be available on the page
  let tries = 0;
  const tryInit = () => {
    tries++;
    if (window.supabase && window.supabase.auth) {
      initAutoTokenBridge();
    } else if (tries < 40) {
      setTimeout(tryInit, 250); // wait up to ~10s
    } else {
      console.warn('[OutreachOS] Supabase not found on web app page — cannot auto-set JWT');
    }
  };

  async function initAutoTokenBridge() {
    try {
      // 1) Get current session and set token immediately if present
      const { data } = await window.supabase.auth.getSession();
      const token = data?.session?.access_token || null;
      if (token) {
        await chrome.storage.local.set({ outreachos_jwt: token });
        console.log('[OutreachOS] JWT set from Supabase session (auto)');
      }

      // 2) React to future auth state changes (login/logout/refresh)
      window.supabase.auth.onAuthStateChange(async (_event, session) => {
        const newToken = session?.access_token || null;
        if (newToken) {
          await chrome.storage.local.set({ outreachos_jwt: newToken });
          console.log('[OutreachOS] JWT refreshed from Supabase (auto)');
        } else {
          await chrome.storage.local.remove('outreachos_jwt');
          console.log('[OutreachOS] JWT cleared (signed out)');
        }
      });

      // 3) (Optional) Auto-pick API base from web app origin
      let apiBase = null;
      if (location.origin.startsWith('http://localhost:3000')) {
        apiBase = 'http://localhost:3000/api/workflow';
      } else if (location.origin.startsWith('https://outreachos.tech')) {
        apiBase = 'https://outreachos.tech/api/workflow';
      }
      if (apiBase) {
        await chrome.storage.local.set({ outreachos_api_base: apiBase });
        console.log('[OutreachOS] API base set (auto):', apiBase);
      }
    } catch (e) {
      console.warn('[OutreachOS] Auto token setup error:', e?.message || e);
    }
  }

  tryInit();
  // --- Resume compose flow (runs on /messaging/compose/ if a resume payload exists) ---
// --- Resume compose flow (runs on /messaging/compose/ if a resume payload exists) ---
(async function resumeComposeIfRequested() {
  try {
    const stored = await chrome.storage.local.get('outreachos_resume_compose');
    const payload = stored && stored.outreachos_resume_compose;
    if (!payload) return;

    // clear resume immediately to avoid duplicates
    await chrome.storage.local.remove('outreachos_resume_compose');

    console.log('[GlobalComposeResume] found resume payload', payload);
    const NAME = String(payload.name || '').trim();
    const BODY = String(payload.body || 'Hi!');
    const sleep = ms => new Promise(r => setTimeout(r, ms));

    // Helper: robust wait-for-selector with timeout
    const waitForSelectorFlexible = (selectors, root = document, timeout = 15000) =>
      new Promise(resolve => {
        const end = Date.now() + timeout;
        const check = () => {
          for (const sel of selectors) {
            try {
              const el = (root || document).querySelector(sel);
              if (el) return resolve(el);
            } catch (e) {}
          }
          if (Date.now() < end) return setTimeout(check, 120);
          resolve(null);
        };
        check();
      });

    // Helper: type into plain <input> or contenteditable reliably
    const typeInto = async (el, text) => {
      if (!el) return false;
      el.focus();
      // If input/textarea, set value char-by-char and fire events
      if (el.tagName === 'INPUT' || el.tagName === 'TEXTAREA') {
        el.value = '';
        for (const ch of String(text || '')) {
          try { document.execCommand('insertText', false, ch); } catch { el.value += ch; }
          el.dispatchEvent(new InputEvent('input', { bubbles: true, data: ch, inputType: 'insertText' }));
          await sleep(15);
        }
        el.dispatchEvent(new Event('change', { bubbles: true }));
        return true;
      }
      // contenteditable
      try {
        // use global helper if present
        if (typeof typeIntoEditor === 'function') {
          await typeIntoEditor(el, String(text || ''));
          return true;
        }
      } catch (e) {}
      // fallback per-char
      el.innerHTML = '';
      for (const ch of String(text || '')) {
        try { document.execCommand('insertText', false, ch); } catch { el.appendChild(document.createTextNode(ch)); }
        el.dispatchEvent(new InputEvent('input', { bubbles: true, data: ch, inputType: 'insertText' }));
        await sleep(15);
      }
      el.dispatchEvent(new Event('change', { bubbles: true }));
      return true;
    };

    // Wait for typeahead input (support multiple selector variants)
    const typeaheadSelectors = [
      'input.msg-connections-typeahead__search-field',
      'input[role="combobox"]',
      'input[name="recipients"]',
      'input[placeholder*="Type a name"]',
      'input[aria-autocomplete="list"]'
    ];
    const typeahead = await waitForSelectorFlexible(typeaheadSelectors, document, 20000);
    if (!typeahead) {
      console.error('[GlobalComposeResume] compose UI not ready (typeahead missing)');
      return;
    }
    console.log('[GlobalComposeResume] found typeahead element ->', typeahead);

    // Clear + type the name slowly (to trigger LinkedIn suggestions)
    try {
      typeahead.focus();
      typeahead.value = '';
      typeahead.dispatchEvent(new Event('input', { bubbles: true }));
      await sleep(120);
      for (const ch of NAME) {
        try { document.execCommand('insertText', false, ch); } catch { typeahead.value += ch; }
        typeahead.dispatchEvent(new InputEvent('input', { bubbles: true, data: ch, inputType: 'insertText' }));
        await sleep(18);
      }
      console.log('[GlobalComposeResume] typed into typeahead:', NAME);
    } catch (e) {
      console.warn('[GlobalComposeResume] failed to type into typeahead with error', e);
    }

    // Wait for suggestion list (supports portal/listbox variants)
    const listRoot = document.querySelector('.msg-connections-typeahead__result-list-container') ||
                     document.querySelector('[role="listbox"]') ||
                     document.querySelector('ul[role="listbox"]');
    if (!listRoot) {
      // give a small extra wait for dynamic mount then re-check
      await sleep(600);
    }
    const listAfter = document.querySelector('.msg-connections-typeahead__result-list-container') ||
                      document.querySelector('[role="listbox"]') ||
                      document.querySelector('ul[role="listbox"]');
    const finalList = listRoot || listAfter;
    if (!finalList) {
      console.error('[GlobalComposeResume] suggestion list not found');
      return;
    }

    const options = Array.from(finalList.querySelectorAll('li[role="option"], [role="option"], li'));
    if (!options.length) {
      console.error('[GlobalComposeResume] no suggestions found for name:', NAME);
      return;
    }

    const norm = s => (s || '').trim().toLowerCase().replace(/\s+/g, ' ');
    const nameKey = norm(NAME);
    let pick = options.find(li => {
      const txt = norm(li.innerText || li.textContent || '');
      return txt === nameKey || txt.includes(nameKey) || nameKey.includes(txt);
    }) || options[0];

    pick.scrollIntoView({ block: 'nearest' });
    await sleep(120);
    const clickTarget = pick.querySelector('button, a, [role="option"]') || pick;
    clickTarget.dispatchEvent(new MouseEvent('mouseover', { bubbles: true }));
    clickTarget.dispatchEvent(new MouseEvent('mousedown', { bubbles: true }));
    clickTarget.dispatchEvent(new MouseEvent('mouseup', { bubbles: true }));
    clickTarget.click();
    console.log('[GlobalComposeResume] clicked suggestion ->', pick.textContent && pick.textContent.trim().slice(0,60));

    await sleep(600);

    // Confirm recipient pill added (multiple variants)
    let pillConfirmed = false;
    const addedSelectors = [
      '.msg-connections-typeahead__added-recipients',
      '.msg-connections-typeahead__added-recipient',
      '.msg-entity-list__item',
      '.msg-connections-typeahead__added-recipients li'
    ];
    for (const sel of addedSelectors) {
      const el = document.querySelector(sel);
      if (el) {
        pillConfirmed = Array.from(el.children || []).some(c => c.tagName !== 'INPUT' && (c.textContent || '').trim());
        if (pillConfirmed) break;
      }
    }
    // final fallback: check if any token with the name exists
    if (!pillConfirmed) {
      const tokens = Array.from(document.querySelectorAll('[role="grid"] span, .msg-pill__text, .msg-pill--recipient'));
      pillConfirmed = tokens.some(t => norm(t.textContent || '').includes(nameKey));
    }

    if (!pillConfirmed) {
      console.error('[GlobalComposeResume] recipient pill not detected after click; aborting');
      return;
    }
    console.log('[GlobalComposeResume] recipient pill confirmed');

    // Find the editor (try several variants)
    const editorSelectors = [
      'div.msg-form__contenteditable[contenteditable="true"]',
      'div.ql-editor[contenteditable="true"]',
      'div.msg-form__contenteditable',
      'textarea[aria-label*="Write a message"]',
      'div[contenteditable="true"]'
    ];
    let editor = null;
    for (const s of editorSelectors) {
      editor = document.querySelector(s);
      if (editor) break;
    }
    // final wait if not immediately found
    if (!editor) editor = await waitForSelectorFlexible(editorSelectors, document, 6000);
    if (!editor) {
      console.error('[GlobalComposeResume] compose editor not found');
      return;
    }
    console.log('[GlobalComposeResume] compose editor found');

    // Type message and send
    await typeInto(editor, BODY);
    await sleep(200);

    // Find send button (variants)
    const sendSelectors = [
      'button.msg-form__send-button',
      'button[aria-label="Send"]',
      'button[data-control-name="send"]',
      'button[type="submit"]'
    ];
    let sendBtn = null;
    for (const s of sendSelectors) {
      sendBtn = document.querySelector(s);
      if (sendBtn) break;
    }
    if (!sendBtn) {
      // try pressing Enter as a fallback
      console.warn('[GlobalComposeResume] send button not found; trying Enter key fallback');
      editor.dispatchEvent(new KeyboardEvent('keydown', { key: 'Enter', code: 'Enter', bubbles: true }));
      editor.dispatchEvent(new KeyboardEvent('keyup', { key: 'Enter', code: 'Enter', bubbles: true }));
      await sleep(300);
      console.log('[GlobalComposeResume] Enter fallback attempted');
      return;
    }

    // If disabled, poke editor then try again
    if (sendBtn.disabled || sendBtn.getAttribute('aria-disabled') === 'true') {
      try { document.execCommand('insertText', false, ' '); } catch { editor.appendChild(document.createTextNode(' ')); }
      editor.dispatchEvent(new InputEvent('input', { bubbles: true, data: ' ', inputType: 'insertText' }));
      await sleep(200);
      // re-resolve sendBtn
      sendBtn = document.querySelector('button.msg-form__send-button') || sendBtn;
    }

    sendBtn.click();
    console.log('[GlobalComposeResume] clicked Send for message:', BODY);

    // Micro-confirmation (best-effort)
    try { await waitForOutgoingBubble(BODY, 2500); console.log('[GlobalComposeResume] outgoing bubble observed — send likely succeeded'); } catch (e) {
      console.warn('[GlobalComposeResume] no outgoing bubble detected (non-blocking)');
    }
  } catch (e) {
    console.error('[GlobalComposeResume] unexpected error:', e);
  }
})();

})();
///lane 2 message send


// Minimal queue worker for OUTREACHOS_OPEN_TYPE_SEND (uses conversation list)
// Minimal queue worker for OUTREACHOS_OPEN_TYPE_SEND (uses conversation list)

function normMsgText(s) {
  return (s || '')
    .replace(/\r/g, '')
    .replace(/\u00A0/g, ' ')
    .replace(/[ \t]+/g, ' ')
    .replace(/\n{2,}/g, '\n')
    .trim();
}
async function handleMessageSending(payload) {
  const { id, name, message } = payload || {};
  console.log(`[Message] Starting to send message to: ${name}`);
  try {
    // Open conversation by name
    const opened = await openConversationByName(name);
    if (!opened) throw new Error('Conversation not found');

    // Let the thread load
    await new Promise(r => setTimeout(r, 2000));

    // Type into the thread editor
    const input = document.querySelector('div.msg-form__contenteditable[contenteditable="true"]');
    if (!input) throw new Error('Message input box not found');

    await simulateKeyboardTyping(input, message);

    // Wait until Send is enabled and visible
    const sendBtn = await waitForSendButtonEnabled();
    if (!sendBtn) throw new Error('Send button not found or disabled');

    // Send
    sendBtn.click();
    console.log('[Message] Successfully sent message to:', name);

    // Store a “just-sent” hint for server reconciliation
    await chrome.storage.local.set({
      outreachos_last_sent_hint: {
        name,
        text_norm: normMsgText(String(message)),
        at: new Date().toISOString()
      }
    });

    return { success: true, messageId: id };
  } catch (err) {
    console.error('[Message] Error sending message:', err);
    return { success: false, error: err.message };
  }
}
/* ==================== MESSAGE QUEUE SYSTEM ====================
   Ensures only one message send operation runs at a time and preserves order.
================================================================= */
let isMessageSending = false;
let messageQueue = [];

/* processMessageQueue: pulls next queued message and sends it (FIFO) */
async function processMessageQueue() {
  if (isMessageSending || messageQueue.length === 0) return;
  isMessageSending = true;
  const messageRequest = messageQueue.shift();
  try {
    const result = await handleMessageSending(messageRequest.payload);
    messageRequest.resolve(result);
  } catch (error) {
    messageRequest.reject(error);
  } finally {
    isMessageSending = false;
    if (messageQueue.length > 0) setTimeout(processMessageQueue, 1000);
  }
}

/* queueMessage: enqueues a message payload and returns a promise for its result */
function queueMessage(payload) {
  return new Promise((resolve, reject) => {
    messageQueue.push({ payload, resolve, reject });
    processMessageQueue();
  });
}

/* ==================== RUNTIME MESSAGE HANDLERS ====================
   Handles messages received from background.js or popup.
==================================================================== */

/* Handshake trigger from popup to fetch Supabase session token if available */
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request && request.action === "OUTREACHOS_TRIGGER_HANDSHAKE") {
    if (window.supabase) {
      window.supabase.auth.getSession().then(({ data }) => {
        const session = data?.session;
        if (session?.access_token) {
          window.postMessage({
            type: "OUTREACHOS_EXTENSION_HANDSHAKE_RESPONSE",
            session: { access_token: session.access_token }
          }, "*");
        }
      });
    }
  }
});

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  // PING: simple liveness check used by background.js before sending actions
  if (request.action === "PING") {
    sendResponse({ alive: true });
    return true; // MUST return true to keep message port open for response
  }

  if (request.type === "SEND_ON_THREAD_PAGE") {
    (async () => {
      DM_LOG('▶ SEND_ON_THREAD_PAGE: Received request');
      await waitForReady(15000);
      const editor = await waitForSelector('div.msg-form__contenteditable[contenteditable="true"]', 10000);
      if (!editor) {
        return sendResponse({ success: false, error: 'editor_not_found_on_thread_page' });
      }

      DM_LOG('├─ ✅ Found editor. Typing message...');
      await typeIntoEditor(editor, request.payload.body);
      await new Promise(r => setTimeout(r, 500));

      const sendBtn = await waitForSelector('button.msg-form__send-button', 3000);
      if (!sendBtn || sendBtn.disabled) {
        return sendResponse({ success: false, error: 'send_button_not_found_on_thread_page' });
      }
      
      sendBtn.click();
      DM_LOG('├─ Waiting 2 seconds after send...');
      await new Promise(r => setTimeout(r, 2000));
      sendResponse({ success: true, evidence: { action_type: 'dm', path: 'direct_to_thread' } });
    })();
    return true;
  }

  // Phase 1: Attempt to run the entire DM workflow silently in the background
  if (request.type === "ATTEMPT_DM_BACKGROUND") {
    (async () => {
      DM_LOG('▶ ATTEMPT_DM_BACKGROUND: Received request');
      // This is a silent attempt, so we use a shorter timeout and expect it might fail.
      const result = await performWorkflowDM(request.payload, { isBackgroundAttempt: true });
      sendResponse(result);
    })();
    return true; 
  }

  // Phase 2: Execute the DM workflow in the foreground as a fallback
  if (request.type === "EXECUTE_DM_FOREGROUND") {
    (async () => {
      DM_LOG('▶ EXECUTE_DM_FOREGROUND: Received request, starting focused workflow.');
      // This is a focused attempt, so we use longer timeouts.
      const result = await performWorkflowDM(request.payload, { isBackgroundAttempt: false });
      sendResponse(result);
    })();
    return true;
  }
});



/* OUTREACHOS_OPEN_TYPE_SEND: enqueue and send a LinkedIn message in correct order */
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "OUTREACHOS_OPEN_TYPE_SEND") {
    console.log('[OUTREACHOS_OPEN_TYPE_SEND] Received request:', request.payload);
    queueMessage(request.payload)
      .then(result => {
        console.log('[OUTREACHOS_OPEN_TYPE_SEND] Sending response:', result);
        sendResponse(result);
      })
      .catch(error => {
        console.error('[OUTREACHOS_OPEN_TYPE_SEND] Error:', error);
        sendResponse({ success: false, error: error.message });
      });
    return true; // keep channel open for async
  }
});

/* OUTREACHOS_OPEN_CONVERSATION: opens a conversation by name or url (helper wrapper) */
chrome.runtime.onMessage.addListener(async (request, sender, sendResponse) => {
  if (request.action === "OUTREACHOS_OPEN_CONVERSATION") {
    try {
      const ok = await openConversationByNameOrUrl(request.payload);
      sendResponse({ success: !!ok });
    } catch (error) {
      sendResponse({ success: false, error: error.message });
    }
    return true;
  }
});

/* OUTREACHOS_SCRAPE_SEARCH: scrapes visible search results and shows modal to save */
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "OUTREACHOS_SCRAPE_SEARCH") {
    const leads = getSearchResults();
    console.log("Scraped leads:", leads);
    if (leads.length === 0) {
      alert("No leads found on this page.");
      return;
    }
    showProspectListModal(leads, (listId, leadCount) => {
      sendLeadsHumanLike(leads, leadCount || 50, listId);
    });
  }
});

/* CHECK_BADGE_COUNT: returns the unread messages badge count on the feed page */
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "CHECK_BADGE_COUNT") {
    if (window.location.href.startsWith('https://www.linkedin.com/feed')) {
      const selectors = [
        '.msg-overlay-bubble-header__unread-count abbr',
        '.msg-overlay-bubble-header__badge-container .notification-badge__count',
        '.msg-overlay-bubble-header__badge .notification-badge__count',
        '.msg-overlay-bubble-header .notification-badge__count',
        '[data-test-messaging-badge] abbr',
        '.msg-overlay-bubble-header__unread-count',
        '.global-nav__primary-link-messaging-count abbr',
        '.global-nav__primary-link-messaging-count'
      ];
      let badgeCount = 0;
      for (const selector of selectors) {
        const element = document.querySelector(selector);
        if (element) {
          const text = element.textContent || element.innerText || '';
          const count = parseInt(text.trim()) || 0;
          if (count > 0) { badgeCount = count; break; }
        }
      }
      sendResponse({ badgeCount });
    } else {
      sendResponse({ badgeCount: 0 });
    }
    return true;
  }
});

/* SCRAPE_CONNECTIONS: scrapes top N "Recently added" connections from Connections page */
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "SCRAPE_CONNECTIONS") {
    (async () => {
      try {
        const n = typeof request.topN === 'number' ? request.topN : 100;
        const res = await scrapeConnectionsTopN(n);
        sendResponse(res);
      } catch (e) {
        sendResponse({ success: false, error: e?.message || 'SCRAPE_CONNECTIONS failed' });
      }
    })();
    return true; // async
  }
});

/* ==================== MESSAGE SENDING ====================
   Pragmatic flow: do NOT close overlays (LinkedIn blocks programmatic close).
   Instead:
   - Click “Message” → capture the new overlay (focus/diff)
   - Temporarily hide other overlays (no pointer, no opacity)
   - Type with real events to enable Send
   - Send from that overlay
   - Unhide others
================================================================= */
// Lane 2 (Workflow): OUTREACHOS_WORKFLOW_DM — Compose overlay with slow waits, keyboard commit, strict verification
/// Lane 2 (Workflow): Reliable DM — profile Message button first, then global compose fallback

/// Lane 2 (Workflow): Reliable DM — profile Message button first, then global compose fallback

// Helpers (scoped to Lane 2 DM)



// Hidden-tab friendly warm-up: jump scrolls (no smooth) + bring actions into view
async function warmUpProfileHiddenSafe() {
  DM_LOG('├─ warmUpProfileHiddenSafe');
  await _warmUpProfileHiddenSafe();
  DM_LOG.exit();
}
async function _warmUpProfileHiddenSafe() {
  const sleep = (ms) => new Promise(r => setTimeout(r, document.hidden ? Math.max(ms, 1200) : ms));
  const jump = (top) => { try { window.scrollTo(0, top); void document.body?.offsetHeight; } catch {} };

  const H = Math.max(document.documentElement?.scrollHeight || 0, document.body?.scrollHeight || 0) || 2000;

  // Deep warm-up passes (avoid rAF-based smooth in hidden tabs)
  jump(0); await sleep(300);
  jump(Math.floor(H * 0.25)); await sleep(700);
  jump(Math.floor(H * 0.60)); await sleep(900);
  jump(H); await sleep(1400);
  jump(0); await sleep(800);

  // Center the actions bar where Message button lives
  const actions = document.querySelector('.pvs-profile-actions') || document.querySelector('#profile-content');
  actions?.scrollIntoView({ behavior: document.hidden ? 'auto' : 'smooth', block: 'center' });
  await sleep(500);
}



function isVisible(el) {
  if (!el) return false;
  const r = el.getBoundingClientRect();
  return r.width > 0 && r.height > 0;
}
function normText(s) {
  return (s || '')
    .replace(/\r/g, '')
    .replace(/\u00A0/g, ' ')
    .replace(/[ \t]+/g, ' ')
    .replace(/\n{2,}/g, '\n')
    .trim();
}
function extractSlugFromUrl(url) {
  try {
    const m = new URL(url).pathname.match(/\/in\/([^/]+)\/?/i);
    return m ? m[1].toLowerCase() : null;
  } catch {
    return null;
  }
}
async function waitForSelector(sel, timeout = 12000, root = document) {
  const t0 = Date.now();
  while (Date.now() - t0 < timeout) {
    const el = root.querySelector(sel);
    if (el) return el;
    await new Promise(r => setTimeout(r, 120));
  }
  return null;
}
async function waitForReadyPage(timeout = 12000) {
  const end = Date.now() + timeout;
  while (Date.now() < end) {
    if ((document.readyState === 'complete' || document.readyState === 'interactive') && document.body) return true;
    await new Promise(r => setTimeout(r, 120));
  }
  return false;
}
async function typeIntoEditor(editor, text) {
  editor.click(); editor.focus();
  let ok = false;
  try { ok = document.execCommand('insertText', false, String(text)); } catch {}
  if (!ok) {
    const s = String(text);
    for (const ch of s) {
      try { document.execCommand('insertText', false, ch); } catch { editor.appendChild(document.createTextNode(ch)); }
      editor.dispatchEvent(new InputEvent('input', { bubbles: true, data: ch, inputType: 'insertText' }));
      await new Promise(r => setTimeout(r, 14));
    }
  }
  editor.dispatchEvent(new Event('change', { bubbles: true }));
}
async function waitForOutgoingBubble(message, timeout = 2000) {
  const want = normText(String(message));
  const start = Date.now();
  while (Date.now() - start < timeout) {
    const nodes = Array.from(document.querySelectorAll('li.msg-s-message-list__event, div.msg-s-message-list__event'));
    const last = nodes[nodes.length - 1];
    if (last) {
      const text = normText((last.innerText || '').trim());
      const fromMe =
        last.className.includes('outgoing') ||
        !!last.querySelector('[data-control-name="outgoing_message"]') ||
        !!last.querySelector('[class*="message-out"]');
      if (fromMe && text === want) return true;
    }
    await new Promise(r => setTimeout(r, 150));
  }
  return false;
}

// Main DM function used by OUTREACHOS_WORKFLOW_DM

async function performWorkflowDM(task = {}, options = { isBackgroundAttempt: false }) {
  DM_LOG('▶ performWorkflowDM', { task, options });
  let res = { success: false, error: 'initial' };
  try {
    // Use a shorter wait for background, longer for foreground
    await waitForReady(options.isBackgroundAttempt ? 15000 : 30000);

    if (!task.name) {
      const h1 = document.querySelector('h1.inline.t-24, h1.eHlIbwKxKZFxwgUEhnMusvqlEbbIQSk, h1');
      if (h1 && (h1.textContent || '').trim()) {
        task.name = h1.textContent.trim();
        DM_LOG("👤 [Workflow DM] Captured profile name:", task.name);
      }
    }

    const targetName = normText(task?.name || '');
    const body = String(task?.body || 'Hi!');
    const slug = extractSlugFromUrl(task?.linkedin_url || location.href) || null;

    res = await tryDirectProfileDM({ targetName, body, slug }, options);
    if (res?.success) {
        DM_LOG('└─ ✅ SUCCESS');
        return res;
    }

    // Only retry if it's a focused, foreground attempt
    if (!options.isBackgroundAttempt) {
      const retriable = new Set([
        'message_button_not_found',
        'overlay_not_found',
        'compose_editor_not_found',
        'send_button_not_found'
      ]);
      if (retriable.has(res?.error)) {
        DM_LOG(`├─ Direct DM failed with retriable error (${res.error}). Retrying once...`);
        res = await tryDirectProfileDM({ targetName, body, slug }, options);
        if (res?.success) {
          DM_LOG('└─ ✅ SUCCESS on retry');
          return res;
        }
        DM_LOG(`├─ Direct DM retry failed (${res.error}).`);
      }
    }
  } catch (e) {
    res = { success: false, error: e.message || 'performWorkflowDM_exception' };
    DM_LOG('└─ ❌ EXCEPTION: ' + res.error);
  }

  if (!res.success) {
    DM_LOG('└─ ❌ FINAL ERROR: ' + res.error);
  }
  return res;
}

// Direct profile overlay path
async function tryDirectProfileDM({ targetName, body, slug }, options = { isBackgroundAttempt: false }) {
  DM_LOG('├─ tryDirectProfileDM', { targetName, slug, options });
  const res = await _tryDirectProfileDM({ targetName, body, slug }, options);
  if (!res.success) DM_LOG('│  └─ ERROR: ' + res.error);
  DM_LOG.exit();
  return res;
}
async function _tryDirectProfileDM({ targetName, body, slug }, options = { isBackgroundAttempt: false }) {
  DM_LOG('│  ├─ Looking for profile Message button...');
  const messageButtonSelector = 'button[aria-label^="Message"][class*="artdeco-button"][class*="primary"], .pvs-profile-actions button[aria-label*="Message"], button[aria-label*="Message"]';
  
  let msgBtn = await waitForSelector(messageButtonSelector, options.isBackgroundAttempt ? 7000 : 12000);

  if (!msgBtn) {
    DM_LOG('│  ├─ Primary selectors failed after wait. Trying fallback by innerText...');
    const byText = Array.from(document.querySelectorAll('button')).find(b => (b.innerText || '').trim().toLowerCase() === 'message');
    if (byText) msgBtn = byText;
  }

  if (!msgBtn) {
    DM_LOG('│  └─ ❌ Message button not found after waiting.');
    return { success: false, error: 'message_button_not_found' };
  }

  DM_LOG('│  ├─ ✅ Found Message button. Clicking...');
  msgBtn.click();
  
  DM_LOG('│  ├─ Pausing for 3 seconds to detect navigation or overlay...');
  await new Promise(r => setTimeout(r, 3000));

  // FORKED PATH: Check if we navigated to a message thread or if an overlay appeared.
  if (window.location.href.includes('/messaging/thread/')) {
    // --- PATH A: We are on the main message thread page ---
    DM_LOG('│  ├─ ✅ Navigated to message thread page.');
    const editor = await waitForSelector('div.msg-form__contenteditable[contenteditable="true"]', 5000);
    if (!editor) {
      DM_LOG('│  └─ ❌ Editor not found on thread page.');
      return { success: false, error: 'editor_not_found_on_thread_page' };
    }

    DM_LOG('│  ├─ ✅ Found editor. Typing message...');
    await typeIntoEditor(editor, body);
    await new Promise(r => setTimeout(r, 500));

    const sendBtn = await waitForSelector('button.msg-form__send-button', 3000);
    if (!sendBtn || sendBtn.disabled) {
      DM_LOG('│  └─ ❌ Send button not found or disabled on thread page.');
      return { success: false, error: 'send_button_not_found_on_thread_page' };
    }
    
    sendBtn.click();
    DM_LOG('│  ├─ Waiting 2 seconds after send...');
    await new Promise(r => setTimeout(r, 2000));
    return { success: true, evidence: { action_type: 'dm', path: 'thread_page' } };

  } else {
    // --- PATH B: We are still on the profile page, look for an overlay ---
    DM_LOG('│  ├─ Still on profile page. Looking for message overlay...');
    const overlay = await waitForSelector('.msg-overlay-conversation-bubble:not(.msg-overlay-conversation-bubble--is-compose)', 5000);

    if (!overlay) {
      DM_LOG('│  └─ ❌ Message overlay not found.');
      return { success: false, error: 'overlay_not_found' };
    }
    DM_LOG('│  ├─ ✅ Found message overlay.');

    const editor = overlay.querySelector('div.msg-form__contenteditable[contenteditable="true"]');
    if (!editor) {
      DM_LOG('│  └─ ❌ Editor not found in overlay.');
      return { success: false, error: 'compose_editor_not_found' };
    }
    DM_LOG('│  ├─ ✅ Found editor. Typing message...');

    await typeIntoEditor(editor, body);
    await new Promise(r => setTimeout(r, 500));

    let sendBtn = overlay.querySelector('button.msg-form__send-button');
    if (!sendBtn || sendBtn.disabled) {
      DM_LOG('│  └─ ❌ Send button not found or disabled in overlay.');
      return { success: false, error: 'send_button_not_found_in_overlay' };
    }
    
    sendBtn.click();
    DM_LOG('│  ├─ Waiting 2 seconds after send...');
    await new Promise(r => setTimeout(r, 2000));
    return { success: true, evidence: { action_type: 'dm', path: 'profile_overlay' } };
  }
}






/* ==================== CONVERSATION OPEN HELPERS ====================
   Support opening a conversation by name (not used by queued send path).
===================================================================== */

/* openConversationByName: finds and opens a conversation in the left list by exact name */
async function openConversationByName(name) {
  const chatList = document.querySelector('ul.msg-conversations-container__conversations-list');
  if (!chatList) {
    console.log('Chat list not found');
    return false;
  }
  const chatItems = chatList.querySelectorAll('li.scaffold-layout__list-item.msg-conversation-listitem');
  for (const item of chatItems) {
    const nameSpan = item.querySelector('span.truncate');
    if (nameSpan && nameSpan.textContent.trim().toLowerCase() === name.trim().toLowerCase()) {
      const link = item.querySelector('.msg-conversation-listitem__link');
      if (link) { link.click(); console.log('Opened conversation:', name); return true; }
    }
  }
  console.warn('Conversation not found for name:', name);
  return false;
}

/* openConversationByNameOrUrl: wrapper that tries to open by URL (future) or by name (now) */
async function openConversationByNameOrUrl(payload) {
  const name = typeof payload === 'string' ? payload : payload?.name;
  if (!name) throw new Error('No name provided');
  return openConversationByName(name);
}

/* ==================== INPUT/CLICK UTILITIES ====================
   Simulated typing (your existing method) and simple helpers.
================================================================= */

/* simulateKeyboardTyping: simulates human typing into a contenteditable input */
async function simulateKeyboardTyping(element, text) {
  element.focus();
  for (const char of text) {
    const keyCode = char.charCodeAt(0);
    element.dispatchEvent(new KeyboardEvent('keydown', { key: char, code: 'Key' + char.toUpperCase(), charCode: keyCode, keyCode, bubbles: true, cancelable: true }));
    element.dispatchEvent(new KeyboardEvent('keypress', { key: char, code: 'Key' + char.toUpperCase(), charCode: keyCode, keyCode, bubbles: true, cancelable: true }));
    document.execCommand('insertText', false, char);
    element.dispatchEvent(new InputEvent('input', { data: char, bubbles: true, cancelable: true }));
    element.dispatchEvent(new KeyboardEvent('keyup', { key: char, code: 'Key' + char.toUpperCase(), charCode: keyCode, keyCode, bubbles: true, cancelable: true }));
    await new Promise(r => setTimeout(r, 50));
  }
}

/* waitForSendButtonEnabled: waits until the send button exists and is visibly enabled */
function waitForSendButtonEnabled(timeout = 10000) {
  return new Promise((resolve) => {
    const start = Date.now();
    const interval = setInterval(() => {
      const sendBtn = document.querySelector('button.msg-form__send-button');
      if (sendBtn && !sendBtn.disabled) {
        const rect = sendBtn.getBoundingClientRect();
        if (rect.width > 0 && rect.height > 0) {
          clearInterval(interval);
          resolve(sendBtn);
        }
      }
      if (Date.now() - start > timeout) {
        clearInterval(interval);
        resolve(null);
      }
    }, 100);
  });
}

/* sendMessage (legacy): simple typing/clicking alternative for compatibility */
async function sendMessage({ message }) {
  const input = document.querySelector('div.msg-form__contenteditable');
  if (!input) throw new Error('Message input box not found');
  input.focus();
  input.innerText = '';
  for (const char of message) {
    document.execCommand('insertText', false, char);
    await new Promise(r => setTimeout(r, 50));
  }
  const sendBtn = document.querySelector('button.msg-form__send-button');
  if (!sendBtn) throw new Error('Send button not found');
  sendBtn.click();
  await new Promise(r => setTimeout(r, 1000));
}

/* ==================== SEARCH RESULTS SCRAPER ====================
   Scrapes search result cards for /in/ profiles with basic fields.
=================================================================== */
function getSearchResults() {
  const leads = [];
  document.querySelectorAll('a[href*="/in/"]').forEach(anchor => {
   const card = anchor.closest('li, div, article'); // broaden match
if (!card) return;
if (card.offsetWidth === 0 && card.offsetHeight === 0) return; // truly invisible only
    const nameSpan = anchor.querySelector('span[aria-hidden="true"]');
    const name = nameSpan ? nameSpan.textContent.trim() : anchor.textContent.trim();
    const link = anchor.href;
    let title = "";
    const titleEl = card.querySelector('div[class*="t-14 t-black t-normal"]');
    if (titleEl) title = titleEl.textContent.trim();
    let location = "";
    const locationEl = card.querySelector('div[class*="t-14 t-normal"]');
    if (locationEl) location = locationEl.textContent.trim();
    let company = "";
    const summaryEl = card.querySelector('p.entity-result__summary--2-lines');
    if (summaryEl) {
      const match = summaryEl.textContent.match(/at (.+)$/);
      if (match) company = match[1].trim();
    }
    const imgEl = card.querySelector('img');
    const profile_image_url = imgEl ? imgEl.src : "";
    if (link && name && !link.includes("/company/") && !leads.some(l => l.linkedin_url === link)) {
      leads.push({ name, title, company, location, linkedin_url: link, profile_image_url, source: "search_results" });
    }
  });
  return leads;
}

/* ==================== WINDOW MESSAGE RELAYS ====================
   Handshake with web app via window.postMessage to store JWT.
================================================================== */

/* Receives Supabase access_token and stores it in chrome.storage.local */
window.addEventListener('message', (event) => {
  if (event.data && event.data.type === 'OUTREACHOS_EXTENSION_HANDSHAKE_RESPONSE') {
    const session = event.data.session;
    if (session?.access_token) {
      chrome.storage.local.set({ outreachos_jwt: session.access_token }, () => {
        window.postMessage('outreachos_linkedin_connected', '*');
      });
    }
  }
});

/* Forwards webapp handshake trigger to background (opens LI tab if needed) */
window.addEventListener('message', (event) => {
  if (event.source !== window) return;
  if (event.data && event.data.type === 'OUTREACHOS_EXTENSION_HANDSHAKE') {
    chrome.runtime.sendMessage({ type: "OUTREACHOS_EXTENSION_HANDSHAKE" }, (response) => {
      if (response?.success) console.log("Handshake relayed to background, tab opened!");
    });
  }
});

/* ==================== GET ENGAGERS UI INJECTION ====================
   Adds a "Get Engagers" button to each feed post to scrape commenters.
===================================================================== */
function injectGetEngagersButtons() {
  document.querySelectorAll('.feed-shared-update-v2').forEach(post => {
    if (post.querySelector('.outreachos-get-engagers-btn')) return;
    const btn = document.createElement('button');
    btn.className = 'outreachos-get-engagers-btn';
    btn.innerText = 'Get Engagers';
    btn.style.background = '#F97316';
    btn.style.color = '#fff';
    btn.style.border = 'none';
    btn.style.borderRadius = '6px';
    btn.style.padding = '4px 12px';
    btn.style.margin = '8px 0 0 8px';
    btn.style.fontWeight = 'bold';
    btn.style.cursor = 'pointer';
    btn.style.fontSize = '14px';
    btn.style.boxShadow = '0 2px 8px rgba(249,115,22,0.10)';
    btn.onclick = function(e) { e.stopPropagation(); showEngagersModal(post); };
    const actionsBar = post.querySelector('.feed-shared-social-actions');
    if (actionsBar) actionsBar.appendChild(btn); else post.appendChild(btn);
  });
}
const observer = new MutationObserver(() => { injectGetEngagersButtons(); });
observer.observe(document.body, { childList: true, subtree: true });
injectGetEngagersButtons();

/* ==================== MODALS (LIST SELECTION, ENGAGERS) ====================
   Minimal UI for selecting list and tracking progress when scraping.
============================================================================= */
function showProspectListModal(leads, onSelectList) {
  document.getElementById('outreachos-modal')?.remove();
  const modal = document.createElement('div');
  modal.id = 'outreachos-modal';
  modal.style.position = 'fixed';
  modal.style.top = '0';
  modal.style.left = '0';
  modal.style.width = '100vw';
  modal.style.height = '100vh';
  modal.style.background = 'rgba(0,0,0,0.35)';
  modal.style.display = 'flex';
  modal.style.alignItems = 'center';
  modal.style.justifyContent = 'center';
  modal.style.zIndex = '999999';
  modal.innerHTML = `
    <div style="background:#181F2A;border-radius:14px;padding:16px 10px;min-width:220px;max-width:320px;box-shadow:0 4px 16px rgba(0,0,0,0.18);display:flex;flex-direction:column;align-items:stretch;border: 2px solid #F97316;">
      <h2 style="color:#F97316; font-size:18px; margin-bottom:12px; text-align:center;">Select Prospect List</h2>
      <select id="outreachos-list-select" style="width:100%;padding:8px;border-radius:6px;margin-bottom:10px;font-size:15px;background:#23283a;color:#fff;border:1px solid #F97316;">
        <option value="">Loading...</option>
      </select>
      <input id="outreachos-list-new" placeholder="Or create new list..." style="width:100%;padding:8px;border-radius:6px;margin-bottom:10px;border:1px solid #F97316;font-size:15px;color:#fff;background:#23283a;" />
      <label style="margin-bottom:6px;font-size:13px;color:#F97316;">How many leads? (max ${leads.length})</label>
      <input id="outreachos-lead-count" type="number" min="1" max="${leads.length}" value="${Math.min(50, leads.length)}" style="width:100%;padding:8px;border-radius:6px;margin-bottom:10px;border:1px solid #F97316;font-size:15px;color:#fff;background:#23283a;" />
      <button id="outreachos-list-confirm" style="background:#F97316;color:#fff;padding:10px 0;width:100%;border:none;border-radius:6px;font-weight:bold;font-size:15px;box-shadow:0 2px 8px rgba(249,115,22,0.10);margin-bottom:6px;">Start Scraping</button>
      <button id="outreachos-list-cancel" style="background:#23283a;color:#fff;padding:8px 0;width:100%;border:none;border-radius:6px;font-size:14px;">Cancel</button>
    </div>
  `;
  document.body.appendChild(modal);
  chrome.storage.local.get(['outreachos_jwt'], async (result) => {
    const jwt = result.outreachos_jwt;
    let lists = [];
    if (jwt) {
      const { SUPABASE_URL, SUPABASE_ANON_KEY } = await getEnv();
      const res = await fetch(`${SUPABASE_URL}/rest/v1/prospect_lists`, {
        headers: {
          apikey: SUPABASE_ANON_KEY,
          Authorization: `Bearer ${jwt}`
        }
      });
      lists = await res.json();
    }
    const select = document.getElementById('outreachos-list-select');
    select.innerHTML = lists.length
      ? lists.map(l => `<option value="${l.id}">${l.name}</option>`).join('')
      : `<option value="">No lists found</option>`;
  });
  document.getElementById('outreachos-list-cancel').onclick = () => modal.remove();
  document.getElementById('outreachos-list-confirm').onclick = async () => {
    const select = document.getElementById('outreachos-list-select');
    const newList = document.getElementById('outreachos-list-new').value.trim();
    const leadCount = Math.min(leads.length, Math.max(1, parseInt(document.getElementById('outreachos-lead-count').value || "50")));
    modal.remove();
    if (newList) {
      try {
        const listId = await createProspectList(newList);
        onSelectList(listId, leadCount);
      } catch (err) {
        alert("Failed to create list: " + err);
      }
    } else {
      onSelectList(select.value, leadCount);
    }
  };
}

/* showEngagersModal: modal to pick a list before scraping commenters from a post */
function showEngagersModal(post) {
  document.getElementById('outreachos-modal')?.remove();
  const modal = document.createElement('div');
  modal.id = 'outreachos-modal';
  modal.style.position = 'fixed';
  modal.style.top = '0';
  modal.style.left = '0';
  modal.style.width = '100vw';
  modal.style.height = '100vh';
  modal.style.background = 'rgba(0,0,0,0.35)';
  modal.style.display = 'flex';
  modal.style.alignItems = 'center';
  modal.style.justifyContent = 'center';
  modal.style.zIndex = '999999';
  modal.innerHTML = `
    <div style="background:#181F2A;border-radius:14px;padding:16px 10px;min-width:220px;max-width:320px;box-shadow:0 4px 16px rgba(0,0,0,0.18);display:flex;flex-direction:column;align-items:stretch;border: 2px solid #F97316;">
      <h2 style="color:#F97316; font-size:18px; margin-bottom:12px; text-align:center;">Scrape Commenters</h2>
      <div style="color:#F97316;font-size:13px;margin-bottom:8px;">Only visible commenters will be scraped. Click 'See more comments' to load all before scraping.</div>
      <select id="outreachos-list-select" style="width:100%;padding:8px;border-radius:6px;margin-bottom:10px;font-size:15px;background:#23283a;color:#fff;border:1px solid #F97316;">
        <option value="">Loading...</option>
      </select>
      <button id="outreachos-engagers-confirm" style="background:#F97316;color:#fff;padding:10px 0;width:100%;border:none;border-radius:6px;font-weight:bold;font-size:15px;box-shadow:0 2px 8px rgba(249,115,22,0.10);margin-bottom:6px;">Start Scraping</button>
      <button id="outreachos-engagers-cancel" style="background:#23283a;color:#fff;padding:8px 0;width:100%;border:none;border-radius:6px;font-size:14px;">Cancel</button>
    </div>
  `;
  document.body.appendChild(modal);
  chrome.storage.local.get(['outreachos_jwt'], async (result) => {
    const jwt = result.outreachos_jwt;
    let lists = [];
    if (jwt) {
      const { SUPABASE_URL, SUPABASE_ANON_KEY } = await getEnv();
      const res = await fetch(`${SUPABASE_URL}/rest/v1/prospect_lists`, {
        headers: {
          apikey: SUPABASE_ANON_KEY,
          Authorization: `Bearer ${jwt}`
        }
      });
      lists = await res.json();
    }
    const select = document.getElementById('outreachos-list-select');
    select.innerHTML = lists.length
      ? lists.map(l => `<option value="${l.id}">${l.name}</option>`).join('')
      : `<option value="">No lists found</option>`;
  });
  document.getElementById('outreachos-engagers-cancel').onclick = () => modal.remove();
  document.getElementById('outreachos-engagers-confirm').onclick = async () => {
    const select = document.getElementById('outreachos-list-select');
    modal.remove();
    scrapeEngagers(post, select.value);
  };
}

/* ==================== ENGAGERS/LEADS SAVE WITH PROGRESS ==================== */
async function scrapeEngagers(post, listId) {
  let people = [];
const anchors = Array.from(post.querySelectorAll('a.comments-comment-meta__description-container'));
anchors.forEach((anchor) => {
const nameEl = anchor.querySelector('.comments-comment-meta__description-title');
const connEl = anchor.querySelector('.comments-comment-meta__data');
const bioEl = anchor.querySelector('.comments-comment-meta__description-subtitle');

const name = (nameEl?.textContent || '').trim();
const connRaw = (connEl?.textContent || '').trim(); // e.g., "• 1st" or "• 2nd"
const levelMatch = connRaw.match(/\b(1st+?|2nd|3rd+?)\b/i);
const level = levelMatch ? levelMatch[1].toLowerCase() : '';
const is_connected = level.startsWith('1st');
const status = is_connected ? 'Connected' : 'New';
const bio = (bioEl?.textContent || '').trim();
const link = anchor.href;

if (link && name && !people.some(p => p.linkedin_url === link)) {
people.push({
name,
title: bio, // bio → title
linkedin_url: link,
source: 'comments', // use "comments" so your filters match
status,
is_connected
});
}
});
  if (people.length === 0) {
    alert("No commenters found. Expand all comments and try again.");
    return;
  }
  await sendEngagersWithProgress(people, listId);
}

async function sendEngagersWithProgress(people, listId) {
  let progressBar = document.createElement('div');
  progressBar.id = 'outreachos-progress-bar';
  progressBar.style.position = 'fixed';
  progressBar.style.bottom = '32px';
  progressBar.style.left = '50%';
  progressBar.style.transform = 'translateX(-50%)';
  progressBar.style.background = '#23283a';
  progressBar.style.borderRadius = '12px';
  progressBar.style.boxShadow = '0 2px 16px rgba(0,0,0,0.12)';
  progressBar.style.padding = '18px 32px';
  progressBar.style.zIndex = '999999';
  progressBar.innerHTML = `
    <div style="font-weight:bold;color:#F97316;margin-bottom:8px;">Scraping commenters...</div>
    <div id="outreachos-progress-text" style="font-size:15px;margin-bottom:8px;">0 / ${people.length}</div>
    <div style="background:#eee;border-radius:8px;height:10px;width:100%;overflow:hidden;">
      <div id="outreachos-progress-inner" style="background:#F97316;height:10px;width:0%;transition:width 0.3s;"></div>
    </div>
    <button id="outreachos-progress-cancel" style="margin-top:12px;background:#eee;color:#333;padding:6px 18px;border:none;border-radius:8px;font-size:14px;cursor:pointer;">Cancel</button>
  `;
  document.body.appendChild(progressBar);

  const sleep = (ms) => new Promise(res => setTimeout(res, ms));
  const updateProgress = (done, total) => {
    const textEl = document.getElementById('outreachos-progress-text');
    const innerEl = document.getElementById('outreachos-progress-inner');
    if (!textEl || !innerEl) return;
    textEl.innerText = `${done} / ${total}`;
    innerEl.style.width = `${Math.round((done / total) * 100)}%`;
  };

  let totalScraped = 0;
  let keepGoing = true;
  const total = people.length;

  document.getElementById('outreachos-progress-cancel').onclick = () => {
    keepGoing = false;
    progressBar.remove();
    alert("Scraping cancelled.");
  };

  // Process in small parallel batches (tune as needed)
  const BATCH_SIZE = 5;
  const BATCH_DELAY_MS = 200;

  for (let i = 0; i < total && keepGoing; i += BATCH_SIZE) {
    const slice = people.slice(i, i + BATCH_SIZE);

    await Promise.all(
      slice.map(p =>
        sendToOutreachOS(p, listId)
          .then(() => {
            totalScraped++;
            updateProgress(totalScraped, total);
          })
          .catch(err => {
            console.error('[Scraper] Failed to save commenter:', err);
            totalScraped++;
            updateProgress(totalScraped, total);
          })
      )
    );

    // small pause between batches
    await sleep(BATCH_DELAY_MS);
  }

  if (document.getElementById('outreachos-progress-bar')) {
    progressBar.remove();
  }
  if (keepGoing) {
    alert(`✅ Done! Scraped and saved ${totalScraped} commenters.`);
  }
}

async function sendLeadsHumanLike(leads, maxLeads, listId) {
  let progressBar = document.createElement('div');
  progressBar.id = 'outreachos-progress-bar';
  progressBar.style.position = 'fixed';
  progressBar.style.bottom = '32px';
  progressBar.style.left = '50%';
  progressBar.style.transform = 'translateX(-50%)';
  progressBar.style.background = '#23283a';
  progressBar.style.borderRadius = '12px';
  progressBar.style.boxShadow = '0 2px 16px rgba(0,0,0,0.12)';
  progressBar.style.padding = '18px 32px';
  progressBar.style.zIndex = '999999';
  progressBar.innerHTML = `
    <div style="font-weight:bold;color:#F97316;margin-bottom:8px;">Scraping leads...</div>
    <div id="outreachos-progress-text" style="font-size:15px;margin-bottom:8px;">0 / ${maxLeads}</div>
    <div style="background:#eee;border-radius:8px;height:10px;width:100%;overflow:hidden;">
      <div id="outreachos-progress-inner" style="background:#F97316;height:10px;width:0%;transition:width 0.3s;"></div>
    </div>
    <button id="outreachos-progress-cancel" style="margin-top:12px;background:#eee;color:#333;padding:6px 18px;border:none;border-radius:8px;font-size:14px;cursor:pointer;">Cancel</button>
  `;
  document.body.appendChild(progressBar);

  const sleep = (ms) => new Promise(res => setTimeout(res, ms));
  const updateProgress = (done, total) => {
    const textEl = document.getElementById('outreachos-progress-text');
    const innerEl = document.getElementById('outreachos-progress-inner');
    if (!textEl || !innerEl) return;
    textEl.innerText = `${done} / ${total}`;
    innerEl.style.width = `${Math.round((done / total) * 100)}%`;
  };

  const totalTarget = Math.min(leads.length, maxLeads);
  let totalScraped = 0;
  let keepGoing = true;

  document.getElementById('outreachos-progress-cancel').onclick = () => {
    keepGoing = false;
    progressBar.remove();
    alert("Scraping cancelled.");
  };

  // Process in small parallel batches (tune as needed)
  const BATCH_SIZE = 5;
  const BATCH_DELAY_MS = 200;

  for (let i = 0; i < totalTarget && keepGoing; i += BATCH_SIZE) {
    const slice = leads.slice(i, i + Math.min(BATCH_SIZE, totalTarget - i));

    await Promise.all(
      slice.map(l =>
        sendToOutreachOS(l, listId)
          .then(() => {
            totalScraped++;
            updateProgress(totalScraped, totalTarget);
          })
          .catch(err => {
            console.error('[Scraper] Failed to save lead:', err);
            totalScraped++;
            updateProgress(totalScraped, totalTarget);
          })
      )
    );

    // small pause between batches
    await sleep(BATCH_DELAY_MS);
  }

  if (document.getElementById('outreachos-progress-bar')) {
    progressBar.remove();
  }
  if (keepGoing) {
    alert(`✅ Done! Scraped and saved ${totalScraped} leads from this page.\nClick 'Next' in LinkedIn and run again for more.`);
  }
}

/* ==================== API HELPERS (LIST + PROSPECT SAVE) ====================
   Talks to Supabase to create lists and save prospects/list-items.
=============================================================================== */
async function createProspectList(name) {
  return new Promise((resolve, reject) => {
    chrome.storage.local.get(['outreachos_jwt'], async (result) => {
      const jwt = result.outreachos_jwt;
      if (!jwt) return reject("Not connected");

      // ✅ always fetch from env instead of assuming globals
      const { SUPABASE_URL, SUPABASE_ANON_KEY } = await getEnv();

      try {
        const res = await fetch(`${SUPABASE_URL}/rest/v1/prospect_lists`, {
          method: "POST",
          headers: {
            apikey: SUPABASE_ANON_KEY,
            Authorization: `Bearer ${jwt}`,
            "Content-Type": "application/json",
            Prefer: "return=representation"
          },
          body: JSON.stringify({ name })
        });

        if (!res.ok) {
          throw new Error(await res.text());
        }

        const data = await res.json();
        if (data && data[0] && data[0].id) {
          resolve(data[0].id);
        } else {
          reject("Failed to create list: " + JSON.stringify(data));
        }
      } catch (err) {
        reject(err.message || err);
      }
    });
  });
}

async function sendToOutreachOS(prospect, listId) {
  return new Promise((resolve, reject) => {
    chrome.storage.local.get(['outreachos_jwt'], async (result) => {
      const jwt = result.outreachos_jwt;
      if (!jwt) {
        alert("Please connect your LinkedIn account to OutreachOS first!");
        reject("Not connected");
        return;
      }

      try {
        const { SUPABASE_URL, SUPABASE_ANON_KEY } = await getEnv();

        // 1) Fetch user id from Supabase auth
        const userResponse = await fetch(`${SUPABASE_URL}/auth/v1/user`, {
          headers: {
            apikey: SUPABASE_ANON_KEY,
            Authorization: `Bearer ${jwt}`
          }
        });
        if (!userResponse.ok) {
          throw new Error(await userResponse.text());
        }
        const userData = await userResponse.json();
        const userId = userData.id;
        if (!userId) { reject("Could not get user ID"); return; }

        // 2) Ensure prospect exists or create it
        let prospectId = null;
        if (prospect.linkedin_url) {
          const checkUrl = `${SUPABASE_URL}/rest/v1/prospects?user_id=eq.${userId}&linkedin_url=eq.${encodeURIComponent(prospect.linkedin_url)}&select=id`;
          const checkResponse = await fetch(checkUrl, {
            headers: {
              apikey: SUPABASE_ANON_KEY,
              Authorization: `Bearer ${jwt}`
            }
          });
          const existingProspects = await checkResponse.json();
          if (existingProspects && existingProspects.length > 0) {
            prospectId = existingProspects[0].id;
          }
        }

        if (!prospectId) {
          const prospectData = {
            name: prospect.name,
            title: prospect.title || null,
            linkedin_url: prospect.linkedin_url || null,
            source: prospect.source || 'extension',
            status: prospect.status || (prospect.is_connected ? 'Connected' : 'New'),
            is_connected: typeof prospect.is_connected === 'boolean' ? prospect.is_connected : false,
            location: prospect.location || null,
            user_id: userId
          };

          const createResponse = await fetch(`${SUPABASE_URL}/rest/v1/prospects`, {
            method: "POST",
            headers: {
              apikey: SUPABASE_ANON_KEY,
              Authorization: `Bearer ${jwt}`,
              "Content-Type": "application/json",
              Prefer: "return=representation"
            },
            body: JSON.stringify(prospectData)
          });
          if (!createResponse.ok) {
            throw new Error(await createResponse.text());
          }
          const newProspect = await createResponse.json();
          if (newProspect && newProspect[0]) prospectId = newProspect[0].id;
          else throw new Error("Failed to create prospect - no ID returned");
        }

        // 3) Attach prospect to list if not already present
        const checkListUrl = `${SUPABASE_URL}/rest/v1/prospect_list_items?prospect_id=eq.${prospectId}&list_id=eq.${listId}&select=id`;
        const checkListResponse = await fetch(checkListUrl, {
          headers: {
            apikey: SUPABASE_ANON_KEY,
            Authorization: `Bearer ${jwt}`
          }
        });
        const existingListItems = await checkListResponse.json();
        if (existingListItems && existingListItems.length > 0) {
          resolve({ success: true, alreadyInList: true });
          return;
        }

        const listItemData = { prospect_id: prospectId, list_id: listId, status: 'new', campaign_id: null };
        const addToListResponse = await fetch(`${SUPABASE_URL}/rest/v1/prospect_list_items`, {
          method: "POST",
          headers: {
            apikey: SUPABASE_ANON_KEY,
            Authorization: `Bearer ${jwt}`,
            "Content-Type": "application/json",
            Prefer: "return=minimal"
          },
          body: JSON.stringify(listItemData)
        });
        if (!addToListResponse.ok) {
          throw new Error(await addToListResponse.text());
        }

        resolve({ success: true });
      } catch (error) {
        console.error('[Scraper] Error:', error);
        reject(error);
      }
    });
  });
}

/* ==================== BADGE WATCHER (FEED PAGE) ====================
   Observes the global nav for unread badge changes and reports to background.
====================================================================== */
if (window.location.href.startsWith('https://www.linkedin.com/feed')) {
  (function watchBadge() {
    let lastBadgeCount = 0;
    let lastBadgeTime = 0;
    let hasCheckedOverlay = false;
    const MIN_BADGE_INTERVAL = 5000;

    const getBadgeCount = () => {
      const selectors = [
        '.msg-overlay-bubble-header__unread-count abbr',
        '.msg-overlay-bubble-header__badge-container .notification-badge__count',
        '.msg-overlay-bubble-header__badge .notification-badge__count',
        '.msg-overlay-bubble-header .notification-badge__count',
        '[data-test-messaging-badge] abbr',
        '.msg-overlay-bubble-header__unread-count',
        '.global-nav__primary-link-messaging-count abbr',
        '.global-nav__primary-link-messaging-count'
      ];
      for (const selector of selectors) {
        const element = document.querySelector(selector);
        if (element) {
          const text = element.textContent || element.innerText || '';
          const count = parseInt(text.trim()) || 0;
          if (count > 0) {
            console.log('[Badge] Found badge with selector:', selector, 'count:', count);
            return count;
          }
        }
      }
      return 0;
    };

    const checkMessagingOverlay = async () => {
      if (hasCheckedOverlay) return;
      hasCheckedOverlay = true;
      const toggleButton = document.querySelector('button[class*="msg-overlay-bubble-header__control"]:has(svg use[href*="chevron"])');
      if (!toggleButton) return;
      const isExpanded = toggleButton.querySelector('use')?.getAttribute('href')?.includes('chevron-down');
      if (!isExpanded) { toggleButton.click(); await new Promise(r => setTimeout(r, 500)); }
      const unreadBadges = document.querySelectorAll('.msg-conversation-card__unread-count .notification-badge__count');
      let overlayUnread = 0;
      unreadBadges.forEach(badge => { const count = parseInt((badge.textContent || '').trim()) || 0; overlayUnread += count; });
      if (!isExpanded && toggleButton) {
        const collapseButton = document.querySelector('button:has(use[href*="chevron-down"])');
        if (collapseButton) collapseButton.click();
      }
      if (overlayUnread > 0) {
        console.log('[Badge] Found', overlayUnread, 'unread in messaging overlay');
        chrome.runtime.sendMessage({ action: 'BADGE_CHANGED', badgeCount: overlayUnread, source: 'messaging-overlay' });
      }
    };

    const checkBadgeChange = () => {
      const currentCount = getBadgeCount();
      const now = Date.now();
      if (currentCount !== lastBadgeCount && currentCount > 0 && (now - lastBadgeTime) >= MIN_BADGE_INTERVAL) {
        console.log('[Badge] Changed:', lastBadgeCount, '→', currentCount);
        lastBadgeCount = currentCount;
        lastBadgeTime = now;
        chrome.runtime.sendMessage({ action: 'BADGE_CHANGED', badgeCount: currentCount }, (response) => {
          if (chrome.runtime.lastError) console.error('[Badge] Failed to send badge change:', chrome.runtime.lastError);
          else console.log('[Badge] Badge change sent to background script');
        });
      } else if (currentCount === 0 && lastBadgeCount > 0) {
        console.log('[Badge] Badge cleared (0 messages)');
        lastBadgeCount = 0;
      }
    };

    const startObserver = () => {
      const headerArea = document.querySelector('.global-nav__primary-items') || document.querySelector('.global-nav') || document.body;
      if (headerArea) {
        const mo = new MutationObserver(() => setTimeout(checkBadgeChange, 100));
        mo.observe(headerArea, { childList: true, subtree: true, characterData: true, attributes: true, attributeFilter: ['class', 'aria-label'] });
        console.log('[Badge] Observer started on feed page');
      }
      setTimeout(() => { checkMessagingOverlay(); }, 3000);
    };

    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', () => setTimeout(startObserver, 2000));
    } else {
      setTimeout(startObserver, 2000);
    }

    checkBadgeChange();
    setInterval(() => { checkBadgeChange(); }, 10000);
  })();
}

/* ==================== LANE 2: CONNECTIONS SCRAPER (TOP N RECENTLY ADDED) ====================
   Provides SCRAPE_CONNECTIONS used by background.js daily runner.
============================================================================================== */

/* normalizeLinkedInUrl: converts profile URLs to canonical https://www.linkedin.com/in/slug/ */
function normalizeLinkedInUrl(url) {
  try {
    const u = new URL(url);
    const m = u.pathname.match(/\/in\/[^/]+/i);
    if (m) return `https://www.linkedin.com${m[0]}/`;
    return (url || '').split('#')[0];
  } catch { return url; }
}

/* txt: safe text getter */
function txt(el) { return (el?.innerText || el?.textContent || '').trim(); }

/* visible: quick visibility check by bounding rect */
function visible(el) {
  if (!el) return false;
  const r = el.getBoundingClientRect();
  return r.width > 0 && r.height > 0;
}

/* trySetSortRecentlyAdded: opens sort menu and selects "Recently added" if available */
async function trySetSortRecentlyAdded() {
  try {
    const sortBtn = Array.from(document.querySelectorAll('button,div[role="button"]')).find(el => /sort/i.test(txt(el)));
    if (sortBtn) {
      sortBtn.click();
      await new Promise(r => setTimeout(r, 300));
      const recent = Array.from(document.querySelectorAll('[role="menuitem"],li,button')).find(el => /recent(ly)? added/i.test(txt(el)));
      if (recent) { recent.click(); await new Promise(r => setTimeout(r, 600)); }
    }
  } catch {}
}

/* getAnchors: returns all visible profile anchors on the connections page */
function getAnchors() {
  return Array.from(document.querySelectorAll('a.app-aware-link[href*="/in/"], a[href*="/in/"]'))
    .filter(a => !/\/company\//i.test(a.href));
}

/* findLoadMoreButton: finds the last visible "Load more"/"Show more" button on the page */
function findLoadMoreButton() {
  const candidates = Array.from(document.querySelectorAll('button, a[role="button"], div[role="button"]'))
    .filter(b => visible(b))
    .filter(b => /^(load more|show more|see more|load more results|show more results|see more results)$/i.test(txt(b)));
  return candidates[candidates.length - 1] || null;
}

/* realClick: emits mouse events then .click() for robust React handlers */
function realClick(el) {
  try {
    el.focus();
    el.dispatchEvent(new MouseEvent('mouseover', { bubbles: true }));
    el.dispatchEvent(new MouseEvent('mousedown', { bubbles: true }));
    el.dispatchEvent(new MouseEvent('mouseup', { bubbles: true }));
    el.click();
    return true;
  } catch { return false; }
}

/* loadEnough: clicks "Load more" in small bursts and scrolls to load at least target anchors */
async function loadEnough(target = 120) {
  let prev = 0, stable = 0;
  for (let i = 0; i < 40; i++) {
    // Burst-click up to 5 times per round
    for (let k = 0; k < 5; k++) {
      const btn = findLoadMoreButton();
      if (!btn) break;
      btn.scrollIntoView({ behavior: 'smooth', block: 'center' });
      await new Promise(r => setTimeout(r, 150));
      realClick(btn);
      await new Promise(r => setTimeout(r, 600));
    }
    // Scroll to bottom to trigger lazy load
    window.scrollTo({ top: document.body.scrollHeight, behavior: 'smooth' });
    await new Promise(r => setTimeout(r, 900));
    const count = getAnchors().length;
    if (count >= target) break;
    if (count === prev) stable++; else stable = 0;
    prev = count;
    if (stable >= 2) { window.scrollTo({ top: 0, behavior: 'smooth' }); await new Promise(r => setTimeout(r, 400)); }
    if (stable >= 4) break;
  }
}

/* collectTopN: reads connection cards and builds a unique list of top N connections */
function collectTopN(n = 100) {
  const seen = new Set();
  const out = [];
  const anchors = getAnchors().filter(visible);
  for (const a of anchors) {
    try {
      const profile_url = normalizeLinkedInUrl(a.href || '');
      if (!profile_url.includes('/in/')) continue;

      const card = a.closest('.mn-connection-card, li.mn-connection-card, li.artdeco-list__item, li.entity-list-item, div.mn-connection-card') || a.parentElement;
      const name =
        txt(card?.querySelector('.mn-connection-card__name')) ||
        (a.getAttribute('aria-label') || '').trim() ||
        txt(a.querySelector('span[aria-hidden="true"]')) ||
        txt(a);
      if (!name) continue;

      const key = profile_url.toLowerCase();
      if (seen.has(key)) continue;
      seen.add(key);

      const dateEl = card?.querySelector('time, .time-badge, span[class*="time"]');
      const connected_date = (dateEl?.getAttribute?.('datetime') || txt(dateEl) || '').trim();

      out.push({
        name,
        profile_url,
        connected_date,
        scraped_at: new Date().toISOString(),
        source: 'connections_page'
      });

      if (out.length >= n) break;
    } catch {}
  }
  return out;
}

/* scrapeConnectionsTopN: orchestrates sort→load→collect on the Connections page */
async function scrapeConnectionsTopN(n = 100) {
  try {
    if (!/linkedin\.com\/mynetwork\/invite-connect\/connections/i.test(location.href)) {
      return { success: false, error: 'Not on connections page' };
    }
    await new Promise(r => setTimeout(r, 800));
    await trySetSortRecentlyAdded();
    await new Promise(r => setTimeout(r, 800));
    await loadEnough(n + 20);
    const connections = collectTopN(n);
    return { success: true, connections };
  } catch (e) {
    return { success: false, error: e?.message || 'scrapeConnectionsTopN failed' };
  }
}

/* ==================== WORKFLOW ACTIONS (VIEW, FOLLOW, CONNECT, DM) ====================
   These respond to OUTREACHOS_WORKFLOW_* messages from background.js.
======================================================================================= */

/* waitForElement: mutation-observer helper to wait for a selector */
function waitForElement(selector, timeout = 10000) {
  return new Promise((resolve) => {
    const el = document.querySelector(selector);
    if (el) return resolve(el);
    const observer = new MutationObserver(() => {
      const el2 = document.querySelector(selector);
      if (el2) { observer.disconnect(); resolve(el2); }
    });
    observer.observe(document.documentElement || document.body, { childList: true, subtree: true });
    setTimeout(() => { observer.disconnect(); resolve(null); }, timeout);
  });
}

/* waitForReady: small wait loop until DOM is interactive/complete */
async function waitForReady(timeoutMs = 10000) {
  DM_LOG('├─ waitForReady (' + timeoutMs + ' ms)');
  const ok = await _waitForReady(timeoutMs);
  if (!ok) DM_LOG('│  └─ TIMEOUT');
  else DM_LOG('│  └─ ready');
  return ok;
}
async function _waitForReady(timeoutMs) {
  const end = Date.now() + timeoutMs;
  while (Date.now() < end) {
    if (document.readyState === 'complete' || document.readyState === 'interactive') {
      if (document.body) return true;
    }
    await new Promise(r => setTimeout(r, 150));
  }
  return false;
}

/* scrollPage: simple scroll to simulate a view/read action */
async function scrollPage() {
  const H = Math.max(document.body.scrollHeight, document.documentElement.scrollHeight);
  window.scrollTo({ top: H * 0.5, behavior: 'smooth' });
  await new Promise(r => setTimeout(r, 800));
  window.scrollTo({ top: H, behavior: 'smooth' });
  await new Promise(r => setTimeout(r, 900));
  window.scrollTo({ top: 0, behavior: 'smooth' });
  await new Promise(r => setTimeout(r, 500));
}

/* clickButtonByText: finds a button by exact text (case-insensitive) and clicks it */
function clickButtonByText(text) {
  const btns = Array.from(document.querySelectorAll('button'));
  const match = btns.find(b => (b.innerText || '').trim().toLowerCase() === text.toLowerCase());
  if (match) { match.click(); return true; }
  return false;
}

/* performView: scrolls the profile to evidence a visit */
async function performView(task = {}) {
  try {
    const start = Date.now();
    await waitForReady(8000);
    await scrollPage();
    const time_on_page_ms = Date.now() - start;
    return { success: true, evidence: { time_on_page_ms, scrolled: true, action_type: 'view' } };
  } catch (e) {
    return { success: false, error: e.message || 'view failed' };
  }
}

/* performFollow: clicks Follow and verifies it switched to Following */
async function performFollow(task = {}) {
  try {
    await waitForReady(8000);
    let followBtn =
      document.querySelector('button[aria-label*="Follow"]') ||
      document.querySelector('button[aria-label*="Seguir"]') ||
      null;

    if (!followBtn) {
      clickButtonByText('Follow');
      await new Promise(r => setTimeout(r, 1200));
      followBtn = document.querySelector('button[aria-label*="Following"]') ? null : document.querySelector('button[aria-label*="Follow"]');
    }
    if (followBtn) followBtn.click();
    await new Promise(r => setTimeout(r, 1800));
once
    const following =
      document.querySelector('button[aria-label*="Following"]') ||
      document.querySelector('button[aria-pressed="true"][aria-label*="Follow"]');

    return { success: !!following, evidence: { button_clicked: true, verified_following: !!following, action_type: 'follow' } };
  } catch (e) {
    return { success: false, error: e.message || 'follow failed' };
  }
}

/* performConnect: clicks Connect/Invite and then Send */
async function performConnect(task = {}) {
    console.log('[performConnect] Starting connect action.');

    // Add robust waiting and scrolling from performView
    try {
        console.log('[performConnect] Waiting for page to be ready...');
        await waitForReady(15000); // Increased timeout
        console.log('[performConnect] Scrolling page to load elements...');
        await scrollPage();
        await new Promise(r => setTimeout(r, 1000)); // Extra delay after scrolling
    } catch (e) {
        console.warn('[performConnect] View action part failed, but continuing to connect. Error:', e.message);
    }

    const LIConnect = (() => {
      const sleep = ms => new Promise(r => setTimeout(r, ms));
      const click = el => {
        if (!el) return false;
        el.scrollIntoView({ block: 'center' });
        el.click();
        return true;
      };

      async function checkStatus() {
        const ctaSection = document.querySelector('.yYFPjbooseoXQykQtmEJJvDOmGrvzERYdJI');
        if (!ctaSection) {
          console.log('❌ CTA section not found');
          return 'unknown';
        }
        const buttons = Array.from(ctaSection.querySelectorAll('button'));
        const pendingBtn = buttons.find(btn => (btn.getAttribute('aria-label') || '').toLowerCase().includes('pending'));
        const connectBtn = buttons.find(btn => (btn.getAttribute('aria-label') || '').toLowerCase().includes('connect') || (btn.getAttribute('aria-label') || '').toLowerCase().includes('invite'));
        if (pendingBtn) return 'pending';
        if (connectBtn) return 'can_connect';
        const moreBtn = buttons.find(btn => (btn.getAttribute('aria-label') || '').toLowerCase().includes('more actions'));
        if (moreBtn) {
          click(moreBtn);
          await sleep(500);
          const dropdownItems = document.querySelectorAll('.artdeco-dropdown__item');
          for (let item of dropdownItems) {
            const combined = ((item.getAttribute('aria-label') || '') + ' ' + (item.textContent || '')).toLowerCase();
            if (combined.includes('remove connection') || combined.includes('remove your connection')) {
              document.body.click(); await sleep(200); return 'connected';
            }
            if (combined.includes('withdraw') || combined.includes('pending')) {
              document.body.click(); await sleep(200); return 'pending';
            }
            if (combined.includes('invite') && combined.includes('connect')) {
              document.body.click(); await sleep(200); return 'can_connect_menu';
            }
          }
          document.body.click(); await sleep(200);
        }
        return 'unknown';
      }

      async function connect() {
        let status = 'unknown';
        for (let i = 0; i < 3; i++) {
            console.log(`🔍 Checking status... (Attempt ${i + 1})`);
            status = await checkStatus();
            console.log(`📊 Status: ${status}`);
            if (status !== 'unknown') {
                break;
            }
            console.log('Status is unknown, waiting 2 seconds to retry...');
            await sleep(2000);
        }


        if (status === 'pending') {
          console.log('connect already sent, pending, update db to connect sent');
          return { success: true, evidence: { new_status: 'connect sent', message: 'Invitation pending' } };
        }
        if (status === 'connected') {
          console.log('already connected, update db to connected');
          return { success: true, evidence: { new_status: 'connected', message: 'Already connected' } };
        }
        if (status === 'can_connect') {
          const ctaSection = document.querySelector('.yYFPjbooseoXQykQtmEJJvDOmGrvzERYdJI');
          const connectBtn = Array.from(ctaSection.querySelectorAll('button')).find(btn => (btn.getAttribute('aria-label') || '').toLowerCase().includes('connect') || (btn.getAttribute('aria-label') || '').toLowerCase().includes('invite'));
          if (connectBtn) {
            console.log('🔗 Clicking Connect button...');
            click(connectBtn);
            await sleep(2000);
            const sendBtn = document.querySelector('button[aria-label*="Send without a note"]');
            if (sendBtn) {
              console.log('📤 Clicking Send without note...');
              click(sendBtn);
              console.log('connect sent, update db to connect sent');
              await sleep(2000);
              return { success: true, evidence: { new_status: 'connect sent', message: 'Invitation sent' } };
            } else {
              return { success: false, error: 'Send button not found' };
            }
          } else {
            return { success: false, error: 'Connect button not found' };
          }
        }
        else if (status === 'can_connect_menu') {
          const ctaSection = document.querySelector('.yYFPjbooseoXQykQtmEJJvDOmGrvzERYdJI');
          const moreBtn = Array.from(ctaSection.querySelectorAll('button')).find(btn => (btn.getAttribute('aria-label') || '').toLowerCase().includes('more actions'));
          if (moreBtn) {
            console.log('🔗 Clicking Connect in More menu...');
            click(moreBtn);
            await sleep(500);
            const dropdownItems = document.querySelectorAll('.artdeco-dropdown__item');
            let connectClicked = false;
            for (let item of dropdownItems) {
              const combined = ((item.getAttribute('aria-label') || '') + ' ' + (item.textContent || '')).toLowerCase();
              if (combined.includes('invite') && combined.includes('connect')) {
                click(item);
                connectClicked = true;
                break;
              }
            }
            if (connectClicked) {
              await sleep(2000);
              const sendBtn = document.querySelector('button[aria-label*="Send without a note"]');
              if (sendBtn) {
                console.log('📤 Clicking Send without note...');
                click(sendBtn);
                console.log('connect sent, update db to connect sent');
                await sleep(2000);
                return { success: true, evidence: { new_status: 'connect sent', message: 'Invitation sent' } };
              } else {
                return { success: false, error: 'Send button not found' };
              }
            } else {
              return { success: false, error: 'Connect button not found in menu' };
            }
          } else {
            return { success: false, error: 'More button not found' };
          }
        }
        else {
          return { success: false, error: 'No connect option available' };
        }
      }
      return { connect };
    })();

    return await LIConnect.connect();
}

/* performMessage: opens Message UI on profile, types and sends the message */
/* Note: There is a duplicate waitForElement here (kept to match your original structure) */
function waitForElement(selector, timeout = 12000) {
  return new Promise((resolve) => {
    const el = document.querySelector(selector);
    if (el) return resolve(el);
    const observer = new MutationObserver(() => {
      const el2 = document.querySelector(selector);
      if (el2) {
        observer.disconnect();
        resolve(el2);
      }
    });
    observer.observe(document.body, { childList: true, subtree: true });
    setTimeout(() => {
      observer.disconnect();
      resolve(null);
    }, timeout);
  });
}

async function performMessage(task = {}) {
  try {
    const body = (task.body || 'Hi!').toString();
    await waitForReady(12000);

    // Wait for Message button
    let msgBtn = await waitForElement('.pvs-profile-actions button[aria-label*="Message"]', 10000)
      || await waitForElement('button[aria-label*="Message"]', 10000);
    if (!msgBtn) {
      if (!clickButtonByText('Message')) throw new Error('Message button not found');
      msgBtn = await waitForElement('.pvs-profile-actions button[aria-label*="Message"]', 5000)
        || await waitForElement('button[aria-label*="Message"]', 5000);
      if (!msgBtn) throw new Error('Message button not found after click');
    }
    msgBtn.click();
    await new Promise(r => setTimeout(r, 800));

    // Wait for input
    const input = await waitForElement('div.msg-form__contenteditable[contenteditable="true"],div.msg-form__contenteditable[role="textbox"]', 8000);
    if (!input) throw new Error('Message input not found');
    input.focus();
    await new Promise(r => setTimeout(r, 400));

    document.execCommand('insertText', false, body);
    await new Promise(r => setTimeout(r, 600));

    // Wait for Send button
    const sendBtn = await waitForElement('button.msg-form__send-button[type="submit"],button[aria-label^="Send"]', 6000);
    if (!sendBtn) throw new Error('Send button not found');
    await new Promise(r => setTimeout(r, 400));

    sendBtn.click();
    await new Promise(r => setTimeout(r, 1200)); // keep this one a bit longer

    return { success: true, evidence: { message_sent: true, message_length: body.length, action_type: 'dm' } };
  } catch (e) {
    return { success: false, error: e.message || 'dm failed' };
  }
}

/* Workflow dispatcher: routes OUTREACHOS_WORKFLOW_* messages to specific handlers */
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (!request || !request.type || !request.type.startsWith('OUTREACHOS_WORKFLOW_')) return;
  (async () => {
    try {
      const action = request.type.replace('OUTREACHOS_WORKFLOW_', '').toLowerCase();
      let result;
      if (action === 'view') result = await performView(request.payload);
      else if (action === 'follow') result = await performFollow(request.payload);
      else if (action === 'connect') result = await performConnect(request.payload);
      else if (action === 'dm') result = await performWorkflowDM(request.payload);
      else result = { success: false, error: 'Unknown workflow action' };
      sendResponse(result);
    } catch (err) {
      sendResponse({ success: false, error: err?.message || 'workflow handler error' });
    }
  })();
  return true; // async response
});



// ========== Post-send verification scraping (incremental/full) ==========
// ===================================================================================================
// 🔥 FIX: Enhanced scrape message handler with better error handling and logging
// This addresses the "Could not establish connection" errors by ensuring proper response handling
// ===================================================================================================
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'OUTREACHOS_INCREMENTAL_SCRAPE' || request.action === 'OUTREACHOS_FULL_SCRAPE') {
    console.log(`🔥 [connect.js] Received ${request.action} request`);
    
    (async () => {
      try {
        const { outreachos_sync_paused, lastStateBeforeScraping, outreachos_last_sent_hint } =
          await new Promise(res => chrome.storage.local.get(['outreachos_sync_paused', 'lastStateBeforeScraping', 'outreachos_last_sent_hint'], res));

        if (outreachos_sync_paused) {
          console.log('[Verify] Scrape paused by background.');
          chrome.runtime.sendMessage({ action: 'MESSAGING_SCRAPE_DONE' });
          sendResponse?.({ skipped: true, reason: 'paused' });
          return;
        }

        if (!location.href.includes('/messaging/')) {
          console.warn('[Verify] Not on messaging page, skipping.');
          chrome.runtime.sendMessage({ action: 'MESSAGING_SCRAPE_DONE' });
          sendResponse?.({ success: false, error: 'Not on messaging page' });
          return;
        }

        const list =
          document.querySelector('ul.msg-conversations-container__conversations-list') ||
          document.querySelector('div.msg-conversations-container__conversations-list') ||
          document.querySelector('.msg-conversation-list');

        if (!list) {
          console.warn('[Verify] Conversation list not found, skipping.');
          chrome.runtime.sendMessage({ action: 'MESSAGING_SCRAPE_DONE' });
          sendResponse?.({ success: false, error: 'Conversation list not found' });
          return;
        }

        const mode = request.action === 'OUTREACHOS_FULL_SCRAPE' ? 'full' : 'incremental';
        const limit = mode === 'full' ? 200 : 20;

        const payload = await scrapeConversationsVerify({
          list,
          limit,
          mode,
          verify_pass: (lastStateBeforeScraping === 'sending_messages'),
          last_sent_hint: outreachos_last_sent_hint || null
        });

        // Forward payload to background → server (/api/sync-messages)
        await new Promise(res => {
          chrome.runtime.sendMessage({ action: 'OUTREACHOS_SYNC_MESSAGES', payload }, () => res());
        });

        // Clear hint after a verify pass to avoid stale matches
        if (payload.verify_pass) {
          try { await chrome.storage.local.remove('outreachos_last_sent_hint'); } catch {}
        }

        // Done
        chrome.runtime.sendMessage({ action: 'MESSAGING_SCRAPE_DONE' });
        sendResponse?.({ success: true, conversations: payload.conversations.length });
      } catch (e) {
        console.error('[Verify] Scrape error:', e);
        chrome.runtime.sendMessage({ action: 'MESSAGING_SCRAPE_DONE' });
        sendResponse?.({ success: false, error: e.message });
      }
    })();
    return true; // async
  }
});

async function scrapeConversationsVerify({ list, limit = 20, mode = 'incremental', verify_pass = false, last_sent_hint = null } = {}) {
  const items = Array
    .from(list.querySelectorAll('li.msg-conversation-listitem, lSi.scaffold-layout__list-item.msg-conversation-listitem, li[role="listitem"]'))
    .slice(0, limit);

const conversations = [];

  for (const li of items) {
    const linkEl = li.querySelector('a.msg-conversation-listitem__link') || li.querySelector('a[href*="/messaging/thread/"]');
    linkEl?.click();
    await new Promise(r => setTimeout(r, 900));

    const title = (document.querySelector('h2.msg-thread__thread-title, h1')?.innerText || '').trim();
    const profileLink = document.querySelector('.msg-thread__link-to-profile, a[href*="/in/"]')?.href || null;
    const threadLink = linkEl?.href || null;

    const messages = scrapeVisibleMessagesVerify().map(m => ({
      ...m,
      text_norm: normMsgText(m.text)
    }));

    messages.sort((a, b) => {
      const ta = a.timestamp ? Date.parse(a.timestamp) : 0;
      const tb = b.timestamp ? Date.parse(b.timestamp) : 0;
      return ta - tb;
    });

    const lastOutgoing = [...messages].reverse().find(m => m.from_me) || null;

    conversations.push({
      name: title,
      profile_url: profileLink,
      conversation_url: threadLink,
      last_outgoing_text: lastOutgoing?.text || null,
      last_outgoing_text_norm: lastOutgoing ? normMsgText(lastOutgoing.text) : null,
      last_outgoing_timestamp: lastOutgoing?.timestamp || null,
      messages
    });

    await new Promise(r => setTimeout(r, 250));
  }

  return {
    mode,
    verify_pass,
    verify_reason: verify_pass ? 'post_send' : 'normal',
    client_hints: { last_sent_hint: last_sent_hint || null },
    scraped_at: new Date().toISOString(),
    conversations
  };
}

function scrapeVisibleMessagesVerify() {
  const nodes = Array.from(document.querySelectorAll('li.msg-s-message-list__event, div.msg-s-message-list__event'));
  const out = [];

  for (const n of nodes) {
    const textEl =
      n.querySelector('.msg-s-event-listitem__body') ||
      n.querySelector('.msg-s-message-list__event-content p') ||
      n.querySelector('p') ||
      n;

    const text = (textEl?.innerText || '').trim();
    if (!text) continue;

    const isOutgoing =
      n.className.includes('outgoing') ||
      !!n.querySelector('[data-control-name="outgoing_message"]') ||
      !!n.querySelector('[class*="message-out"]');

    const timeEl = n.querySelector('time[datetime], time, abbr[aria-label]');
    const ts = timeEl?.getAttribute?.('datetime') || timeEl?.getAttribute?.('aria-label') || null;

    out.push({ text, from_me: !!isOutgoing, timestamp: ts || null });
  }

  return out;
}


chrome.runtime.onMessage.addListener((req, sender, sendResponse) => {
  if (req.action === 'FOCUS_GUARD_BEGIN') {
    chrome.tabs.update(sender.tab.id, {active:true}, () => sendResponse({ok:true, token:Date.now()}));
    return true;
  }
  if (req.action === 'FOCUS_GUARD_END' && req.token) {
    chrome.tabs.query({active:true, pinned:false}, tabs => {
      if (tabs[0]) chrome.tabs.update(tabs[0].id, {active:true});
    });
  }
});
/* ==================== FINAL INIT LOG ==================== */
console.log('[OutreachOS] connect.js loaded - ready for state machine coordination');