console.log("OutreachOS background.js loaded (service worker active)");


// ==================== VERSION CONTROL (Supabase-driven kill switch) ====================
const CURRENT_VERSION = chrome.runtime.getManifest().version;

function isOlderVersion(current, min) {
  const c = current.split('.').map(Number);
  const m = min.split('.').map(Number);
  for (let i = 0; i < Math.max(c.length, m.length); i++) {
    if ((c[i] || 0) < (m[i] || 0)) return true;
    if ((c[i] || 0) > (m[i] || 0)) return false;
  }
  return false; // equal
}

async function setupVersionSubscription() {
  const client = await ensureSupabaseAuth();
  if (!client) return;

  client.channel('extension-config')
    .on(
      'postgres_changes',
      { event: '*', schema: 'public', table: 'extension_config' },
      async (payload) => {
        const minVersion = payload.new?.value;
        if (!minVersion) return;

        console.log('DEBUG version event:', payload);

        if (isOlderVersion(CURRENT_VERSION, minVersion)) {
          chrome.notifications.create('outreachos-update-required', {
            type: "basic",
            iconUrl: "icons/icon-128.png",
            title: "Update Required",
            message: `Please update OutreachOS extension to v${minVersion}+ (current v${CURRENT_VERSION})`
          });

          console.warn(`Outdated extension v${CURRENT_VERSION} < required v${minVersion}. Blocking scraping/workflow...`);

          // Persist local flag so popup/content can react
          await chrome.storage.local.set({
            outreachos_blocked: { reason: 'min_version', required: minVersion, current: CURRENT_VERSION, at: Date.now() }
          });

          // Stop workflow alarm only; keep Realtime so sending can still work
          chrome.alarms.clear('workflowAlarm');
        } else {
          // Unblock if requirement went down
          try { await chrome.storage.local.remove('outreachos_blocked'); } catch {}
        }
      }
    )
    .subscribe((status) => {
      console.log("🔥 Version config channel status:", status);
    });
}
// Run once at boot to catch outdated builds even before realtime events arrive
async function initialMinVersionCheck() {
  const client = await ensureSupabaseAuth();
  if (!client) return;

  const { data, error } = await client
    .from('extension_config')
    .select('value')
    .eq('key','min_version')
    .single();

  if (error) {
    console.warn('Version check query failed:', error.message);
    return;
  }

  if (data && isOlderVersion(CURRENT_VERSION, data.value)) {
    chrome.notifications.create('outreachos-update-required', {
      type: "basic",
      iconUrl: "icons/icon-128.png",
      title: "Update Required",
      message: `Please update OutreachOS to v${data.value}+ (current v${CURRENT_VERSION})`
    });

    await chrome.storage.local.set({
      outreachos_blocked: { reason: 'min_version', required: data.value, current: CURRENT_VERSION, at: Date.now() }
    });

    console.warn(`Outdated extension v${CURRENT_VERSION} < required v${data.value}. Blocking scraping/workflow now...`);
    chrome.alarms.clear('workflowAlarm');
  } else {
    try { await chrome.storage.local.remove('outreachos_blocked'); } catch {}
  }
}

// ==================== SUPABASE DIRECT CLIENT (import works now, thanks to type=module) ====================
import { createClient } from 'https://cdn.jsdelivr.net/npm/@supabase/supabase-js/+esm';

// ===================================================================================================
// 🔥 FIX: Environment-aware URL helper to consolidate API endpoints and reduce duplicate calls
// This matches the same helper added to messaging-sync.js for consistency
// ===================================================================================================
const getApiUrl = (endpoint) => {
  // Auto-detect environment - background scripts don't have window.location, so use SUPABASE_URL
  const baseUrl = SUPABASE_URL.includes('localhost')
    ? 'http://localhost:3000/api'
    : 'https://outreachos.tech/api';
  return `${baseUrl}/${endpoint}`;
};

// 🔥 add these two lines
const SUPABASE_URL = 'https://api.outreachos.tech';
let supabase;
let currentJwt = null;
async function ensureSupabaseAuth() {
  const { outreachos_jwt } = await chrome.storage.local.get('outreachos_jwt');
  if (!outreachos_jwt) {
    console.warn('🔥 No JWT for Supabase yet');
    return null;
  }

  // IMPORTANT: recreate client when JWT changes so all requests carry Authorization
  if (!supabase || currentJwt !== outreachos_jwt) {
    const SUPABASE_ANON_KEY = 'eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyAgCiAgICAicm9sZSI6ICJhbm9uIiwKICAgICJpc3MiOiAic3VwYWJhc2UtZGVtbyIsCiAgICAiaWF0IjogMTY0MTc2OTIwMCwKICAgICJleHAiOiAxNzk5NTM1NjAwCn0.dc_X5iR_VP_qT0zsiyj_I_OZ2T9FtRU2BBNWN8Bu4GE'; // replace with your actual anon key
    supabase = createClient(SUPABASE_URL, SUPABASE_ANON_KEY, {
      global: {
        headers: { Authorization: `Bearer ${outreachos_jwt}` }
      },
      auth: {
        persistSession: false,
        autoRefreshToken: false,
        detectSessionInUrl: false
      }
    });
    currentJwt = outreachos_jwt;

    // Ensure Realtime socket uses this JWT
    supabase.realtime.setAuth(outreachos_jwt);
  }

  // Do not rely on session; fetch user from the token directly
  const { data: { user }, error: userErr } = await supabase.auth.getUser(outreachos_jwt);
  console.log('DEBUG auth user:', user?.id || null, userErr?.message || '');


//notification expiry

    if (userErr?.message && userErr.message.toLowerCase().includes('expired')) {
    chrome.notifications.create('outreachos-jwt-expired', {
      type: "basic",
      iconUrl: "icons/icon-128.png",
      title: "OutreachOS: Please Reconnect",
      message: "Your session expired. Open the OutreachOS web app to reconnect."
    });
    refreshJwtFromWebApp();
    setTimeout(() => {
      chrome.runtime.reload();
    }, 10 * 60 * 1000); // reload after 10 minutes
  }
  return supabase;
}



// ==================== CONFIGURATION & CONSTANTS ====================
const LINKEDIN_MESSAGING_URL = "https://www.linkedin.com/messaging/";
const FEED_URL = 'https://www.linkedin.com/feed';
let mainTabId = null; // The single tab we use for all LinkedIn operations

// Cooldown for "No JWT" notification (prevents repeated popups)
const JWT_NOTIFY_COOLDOWN_MS = 5 * 60 * 1000; // 5 minutes

// State machine states - controls the entire extension flow
const STATES = {
  STARTUP_SCRAPING: 'startup_scraping',    // Initial state: checking if first sync is done
  SENDING_MESSAGES: 'sending_messages',     // Sending queued messages from outbox
  INCREMENTAL_SCRAPING: 'incremental_scraping', // Scraping new messages after badge change or message sending
  FEED_WATCHING: 'feed_watching'            // Watching LinkedIn feed for badge changes (idle state)
};

// State management variables
let currentState = STATES.STARTUP_SCRAPING;  // Start in startup mode
let isProcessingOutbox = false;              // Prevents concurrent message sending
let isStateTransitioning = false; // Prevents concurrent state changes
let checkMessagesTimeout = null; 
let drainRetryTimer = null; // One-shot retry when drain is deferred// Timer for checking outbox messages
// Buffer realtime/backlog until we're idle
let outboxEventBuffer = [];
let backlogDetected = false;

function maybeDrainOutboxQueueIfIdle(reason = 'unknown') {
  const idle = currentState === STATES.FEED_WATCHING && !isStateTransitioning && !isProcessingOutbox;

  if (idle) {
    if (drainRetryTimer) {
      clearTimeout(drainRetryTimer);
      drainRetryTimer = null;
    }
    if (backlogDetected || outboxEventBuffer.length > 0) {
      console.log('DEBUG drain: starting send', { reason, backlogDetected, buffered: outboxEventBuffer.length });
      backlogDetected = false;
      outboxEventBuffer.splice(0, outboxEventBuffer.length);
      transitionToSendingMessages();
    } else {
      // Always schedule a keep-alive drain in case a new row appears after 0 backlog
 if (!drainRetryTimer) {
  drainRetryTimer = setTimeout(() => {
    drainRetryTimer = null;
    maybeDrainOutboxQueueIfIdle('keepalive');
  }, 2000);
}
if (reason !== 'keepalive' || Math.random() < 0.02) { // log keepalive only ~once per minute
  console.log('DEBUG drain: no backlog', { reason });
}
    }
  } else {
    console.log('DEBUG drain: deferred', {
      reason,
      state: currentState,
      isStateTransitioning,
      isProcessingOutbox
    });
    if ((backlogDetected || outboxEventBuffer.length > 0) && !drainRetryTimer) {
      drainRetryTimer = setTimeout(() => {
        drainRetryTimer = null;
        maybeDrainOutboxQueueIfIdle('deferred_retry');
      }, 600);
    }
  }
}

/* ==================== PHASE 1: BACKOFF + JITTER HELPERS ==================== */

// Outbox poll backoff config
const OUTBOX_BACKOFF = {
  FAST_MS: 5000,              // 5s when active
  IDLE_MIN_MS: 15000,         // 15s when idle
  IDLE_MAX_MS: 30000,         // 30s when idle
  JITTER_PCT: 0.2,            // ±20% jitter
  NO_CHANGE_THRESHOLD: 3      // After 3 idle checks, switch to idle backoff
};

let outboxNoChangeStreak = 0;     // consecutive idle polls
let isOutboxCheckInFlight = false;

// Workflow claim backoff config (aligns with "every 3–6 min" default)
const WORKFLOW_BACKOFF = {
  BASE_MIN_MS: 3 * 60 * 1000,            // 3 minutes
  BASE_MAX_MS: 6 * 60 * 1000,            // 6 minutes
  ACTIVE_FAST_MS: 60 * 1000,             // 1 minute after a task is found (drain quickly)
  STEPS_MS: [15 * 60 * 1000, 60 * 60 * 1000], // escalate: 15m → 60m after repeated "no task"
  JITTER_PCT: 0.2,                       // ±20%
  NO_TASK_THRESHOLD: 3,                  // 3 consecutive "no task" before escalating to 15m
  MAX_MS: 60 * 60 * 1000                 // safety cap
};

let workflowNoTaskStreak = 0;
let workflowBackoffIndex = -1;           // -1 = base (3–6m cadence), 0..n = STEPS index
let nextWorkflowClaimAt = 0;             // timestamp when we are allowed to claim again

function jitterMs(ms, pct = 0.2) {
  const f = 1 + (Math.random() * 2 * pct - pct);
  return Math.max(0, Math.round(ms * f));
}

function scheduleNextOutboxCheck() {
  console.log('🔥 Poll scheduling disabled — using Realtime subscription');
}

// ==================== 🔥 NEW: WORKFLOW AUTOMATION SYSTEM ====================
// 🔥 NEW: These variables control the workflow automation system that runs parallel to messaging
const WORKFLOW_API_BASE = 'https://outreachos.tech/api/workflow'; // API endpoints for workflow tasks
let workflowQueue = []; // Local retry queue for failed workflow tasks
let isWorkflowActive = false; // Prevents concurrent workflow operations

// ==================== SESSION TRACKING ====================
let sessionStartTime = new Date(); // Track when extension session started

// Also update when JWT is found (user becomes active)
async function updateSessionOnJWT() {
  const { outreachos_jwt } = await chrome.storage.local.get('outreachos_jwt');
  if (outreachos_jwt) {
    sessionStartTime = new Date();
    console.log('🔥 JWT found - session updated:', sessionStartTime);
  }
}

/* ==================== ✅ FIX #1: RESET JWT NOTIFY COOLDOWN ON BOOT ==================== */
async function resetJwtNotifyCooldownOnBoot() {
  try {
    await chrome.storage.local.remove('outreachos_last_jwt_notify_ts');
    console.log('🔥 JWT notify cooldown reset (boot/enable/reload)');
  } catch (e) {
    console.log('JWT cooldown reset error:', e?.message || e);
  }
}

/* ==================== ✅ FIX #2: REUSE EXISTING PINNED LINKEDIN TAB ON BOOT ==================== */
function initMainTabOnBoot() {
  try {
    chrome.tabs.query({}, (tabs) => {
      const existingPinnedTab = tabs.find(
        (tab) => tab.pinned && tab.url && tab.url.startsWith('https://www.linkedin.com/')
      );
      if (existingPinnedTab) {
        console.log('🔥 Boot: reusing existing pinned LinkedIn tab:', existingPinnedTab.id);
        mainTabId = existingPinnedTab.id;
        chrome.storage.local.set({ outreachos_main_tab_id: existingPinnedTab.id });
      }
    });
  } catch (e) {
    console.log('initMainTabOnBoot error:', e?.message || e);
  }
}

// Run both fixes immediately on service worker boot
resetJwtNotifyCooldownOnBoot();
initMainTabOnBoot();

// Ensure workflow alarm exists even on simple enable/reload (not just onInstalled/onStartup)
chrome.alarms.create('workflowAlarm', { periodInMinutes: 1 });
console.log('🔥 workflowAlarm created (boot)');
loadWorkflowQueue(); // safe to call on boot

// ==================== TAB MANAGEMENT ====================
// These functions ensure we always have one LinkedIn tab for all operations

chrome.runtime.onInstalled.addListener(() => {
  console.log('🔥 Extension installed/reloaded - checking for existing pinned LinkedIn tab');
  sessionStartTime = new Date(); // Update session on install

  // Reset JWT notify cooldown on install/reload
  resetJwtNotifyCooldownOnBoot();
  
  // Look for existing PINNED LinkedIn tab (any linkedin.com URL)
  chrome.tabs.query({}, (tabs) => {
    const existingPinnedTab = tabs.find(tab => 
      tab.pinned && tab.url && tab.url.startsWith('https://www.linkedin.com/')
    );
    
    if (existingPinnedTab) {
      console.log('🔥 Found existing pinned LinkedIn tab - reusing:', existingPinnedTab.id);
      mainTabId = existingPinnedTab.id;
      chrome.storage.local.set({ outreachos_main_tab_id: existingPinnedTab.id });
      // No reload needed - content script already working
    } else {
      console.log('🔥 No existing pinned LinkedIn tab found - creating new pinned tab');
      openMainTab();
    }
  });
  
  // Initialize workflow system - creates alarm that runs every 60 seconds
  chrome.alarms.create('workflowAlarm', { periodInMinutes: 1 });
  loadWorkflowQueue(); // Load any saved workflow tasks from storage
});

chrome.runtime.onStartup.addListener(() => {
  console.log('🔥 Browser started - checking for existing pinned LinkedIn tab');
  sessionStartTime = new Date(); // Update session on startup

  // Reset JWT notify cooldown on browser startup
  resetJwtNotifyCooldownOnBoot();
  
  // Initialize workflow system on browser startup
  chrome.alarms.create('workflowAlarm', { periodInMinutes: 1 });
  loadWorkflowQueue(); // Restore workflow queue from storage
  
  chrome.tabs.query({}, (tabs) => {
    // Look for existing PINNED LinkedIn tab
    const existingPinnedTab = tabs.find(tab => 
      tab.pinned && tab.url && tab.url.startsWith('https://www.linkedin.com/')
    );
    if (existingPinnedTab) {
      console.log('🔥 Found existing pinned LinkedIn tab - reusing:', existingPinnedTab.id);
      mainTabId = existingPinnedTab.id;
      chrome.storage.local.set({ outreachos_main_tab_id: existingPinnedTab.id });
      // No reload needed
    } else {
      console.log('🔥 No existing pinned LinkedIn tab found - creating new pinned tab');
      openMainTab();
    }
  });
});

// Creates and pins the main LinkedIn tab at position 0
function openMainTab() {
  chrome.tabs.create({ url: LINKEDIN_MESSAGING_URL, active: false }, (tab) => {
    if (chrome.runtime.lastError) {
      console.error('🔥 Failed to create tab:', chrome.runtime.lastError);
      return;
    }
    
    // Move tab to first position and pin it
    chrome.tabs.move(tab.id, { index: 0 }, (movedTab) => {
      chrome.tabs.update(movedTab.id, { pinned: true }, (pinnedTab) => {
        chrome.storage.local.set({ outreachos_main_tab_id: pinnedTab.id });
        mainTabId = pinnedTab.id;
        console.log('🔥 Main LinkedIn tab created and pinned:', pinnedTab.id);
      });
    });
  });
}

// Ensures the main tab exists before performing operations
async function ensureMainTabExists() {
  if (!mainTabId) {
    console.log('🔥 No main tab ID - creating new tab');
    openMainTab();
    await delay(3000); // Wait for tab creation
    return false;
  }
  
  try {
    // Check if tab still exists
    await chrome.tabs.get(mainTabId);
    return true;
  } catch (error) {
    console.log('🔥 Main tab lost - creating new one');
    openMainTab();
    await delay(3000);
    return false;
  }
}

// ==================== 🔥 NEW: JWT NOTIFICATION SYSTEM ====================
// 🔥 NEW: This function checks if user has connected their JWT token
// If no JWT found, shows a notification to remind user to connect
async function checkJWTAndNotify() {
  const { outreachos_jwt } = await chrome.storage.local.get('outreachos_jwt');
  if (!outreachos_jwt) {
    // Throttled notification
    await maybeNotifyNoJWT('missing_jwt');
    console.log('🔥 JWT missing - notification (throttled) considered');
    return false;
  }
  return true;
}

// Throttled "No JWT" notifier to avoid repeated popups
async function maybeNotifyNoJWT(source = 'missing_jwt') {
  try {
    const now = Date.now();
    const { outreachos_last_jwt_notify_ts } = await chrome.storage.local.get('outreachos_last_jwt_notify_ts');
    if (!outreachos_last_jwt_notify_ts || (now - outreachos_last_jwt_notify_ts) >= JWT_NOTIFY_COOLDOWN_MS) {
      chrome.notifications.create('outreachos-no-jwt', {
        type: "basic",
        iconUrl: "icons/icon-128.png",
        title: "OutreachOS Not Connected",
        message: "Please connect using the OutreachOS extension to enable automation."
      });
      await chrome.storage.local.set({ outreachos_last_jwt_notify_ts: now });
      console.log(`🔥 JWT notify shown (${source})`);
    } else {
      const remaining = JWT_NOTIFY_COOLDOWN_MS - (now - outreachos_last_jwt_notify_ts);
      console.log(`🔥 Skipping JWT notify (${source}) — cooldown ${Math.ceil(remaining / 1000)}s left`);
    }
  } catch (e) {
    console.log('JWT notify error:', e?.message || e);
  }
}
// ==================== STATE MACHINE FUNCTIONS ====================
// These functions control the main flow of the extension

// STARTUP STATE: Check if first sync is done, then decide next action
async function startupFlow() {
  // Prevent multiple startup flows from running
  if (currentState !== STATES.STARTUP_SCRAPING || isStateTransitioning) {
    console.log('🔥 Startup flow already running or wrong state');
    return;
  }
  
  console.log('🔥 STATE: STARTUP_SCRAPING - Checking first sync status');
  
  // Wait for JWT token to be available
  const { outreachos_jwt } = await chrome.storage.local.get('outreachos_jwt');
  if (!outreachos_jwt) {
    await checkJWTAndNotify(); // shows the toast
    setTimeout(startupFlow, 2000);
    return;
  }
  // 🔥 NEW: Update session time when JWT is available
  await updateSessionOnJWT();

  try {
    // Check if first sync has been completed
    // ===================================================================================================
    // 🔥 FIX: Updated to use environment-aware URL helper instead of hardcoded production URL
    // Also improved error handling for more reliable first scrape initialization
    // ===================================================================================================
    const resp = await fetch(getApiUrl('get-sync-status'), {
      headers: { Authorization: `Bearer ${outreachos_jwt}` }
    });
    if (resp.status === 401 || resp.status === 403) {
      await maybeNotifyNoJWT('invalid_jwt_startup');
      await chrome.storage.local.remove('outreachos_jwt');
      setTimeout(startupFlow, 2000);
      return;
    }
    const { first_sync_done } = await resp.json();

    if (!first_sync_done) {
    console.log('🔥 First sync not done - starting full scrape');
await navigateToMessaging();

// ===================================================================================================
// 🔥 FIX: Improved first scrape reliability with better retry logic and fallback handling
// This addresses the issue where first scrape would fail due to content script timing issues
// ===================================================================================================
const ready = await ensureConnectScriptReady(mainTabId, 20000);
if (!ready) {
  console.log('🔥 Content script not ready for FULL_SCRAPE — retrying once before fallback');
  // ✅ FEATURE ENHANCED: Add one retry attempt before falling back
  await delay(3000);
  const retryReady = await ensureConnectScriptReady(mainTabId, 15000);
  
  if (!retryReady) {
    console.log('🔥 Content script still not ready after retry — marking first_sync_done and switching to incremental');
    await markFirstSyncDone();
    await transitionToIncrementalScraping();
  } else {
    // ✅ FEATURE PRESERVED: Retry succeeded, proceed with full scrape
    try {
      await chrome.tabs.sendMessage(mainTabId, { action: 'OUTREACHOS_FULL_SCRAPE' });
    } catch (error) {
      console.log('🔥 FULL_SCRAPE send failed after retry — marking first_sync_done:', error?.message || error);
      await markFirstSyncDone();
      await transitionToIncrementalScraping();
    }
  }
} else {
  // ✅ FEATURE PRESERVED: Content script ready on first try
  try {
    await chrome.tabs.sendMessage(mainTabId, { action: 'OUTREACHOS_FULL_SCRAPE' });
  } catch (error) {
    console.log('🔥 FULL_SCRAPE send failed — marking first_sync_done and switching to incremental:', error?.message || error);
    await markFirstSyncDone();
    await transitionToIncrementalScraping();
  }
}
      // Will transition to feed watching after full scrape completes
    } else {
      console.log('🔥 First sync already done - starting incremental scrape then feed watching');
      await transitionToIncrementalScraping();
    }
  } catch (error) {
    console.error('🔥 Startup error:', error);
    setTimeout(startupFlow, 5000); // Retry on error
  }
}

// ===================================================================================================
// 🔥 FIX: Enhanced incremental scraping with better error handling and content script readiness
// This fixes the "Could not establish connection. Receiving end does not exist" error
// ===================================================================================================
async function transitionToIncrementalScraping() {
  // Blocked? Don't run scrapes
  const { outreachos_blocked } = await chrome.storage.local.get('outreachos_blocked');
  if (outreachos_blocked) {
    console.log('🔥 Incremental scrape blocked by min_version — skipping');
    return;
  }

  if (isStateTransitioning) {
    console.log('🔥 Already transitioning - skipping incremental scraping');
    return;
  }

  isStateTransitioning = true;
  currentState = STATES.INCREMENTAL_SCRAPING;
  console.log('🔥 STATE: INCREMENTAL_SCRAPING - Scraping recent messages');

  if (checkMessagesTimeout) {
    clearTimeout(checkMessagesTimeout);
    checkMessagesTimeout = null;
  }

  try {
    await navigateToMessaging();
    console.log('🔥 Waiting for page to load and content scripts to be ready...');
    
    // ✅ FIX: Wait longer and ensure content script is ready before sending message
    await delay(5000); // Increased from 3000ms to 5000ms
    
    // ✅ FIX: Ensure content script is ready before sending INCREMENTAL_SCRAPE
    const ready = await ensureConnectScriptReady(mainTabId, 15000);
    if (!ready) {
      console.log('🔥 Content script not ready for INCREMENTAL_SCRAPE — skipping to feed watching');
      isStateTransitioning = false;
      await transitionToFeedWatching();
      return;
    }
    
    // ✅ FIX: Wrap message sending in try-catch with specific error handling
    try {
      await chrome.tabs.sendMessage(mainTabId, { action: 'OUTREACHOS_INCREMENTAL_SCRAPE' });
      console.log('🔥 INCREMENTAL_SCRAPE message sent successfully');
    } catch (messageError) {
      console.log('🔥 INCREMENTAL_SCRAPE message failed (content script not responding):', messageError?.message || messageError);
      // If message sending fails, transition to feed watching anyway (graceful degradation)
      isStateTransitioning = false;
      await transitionToFeedWatching();
      return;
    }
  } catch (error) {
    console.error('🔥 Error in incremental scraping:', error);
    isStateTransitioning = false;
    maybeDrainOutboxQueueIfIdle('feed_watch_ready');
    setTimeout(() => maybeDrainOutboxQueueIfIdle('feed_watch_ready_delay'), 400);
    await transitionToFeedWatching();
  }
}

// FEED WATCHING STATE: Monitor LinkedIn feed for badge changes (idle state)
async function transitionToFeedWatching() {
  // Allow transition from incremental scraping, but prevent other concurrent transitions
  if (isStateTransitioning && currentState !== STATES.INCREMENTAL_SCRAPING) {
    console.log('🔥 Already transitioning - skipping feed watching');
    return;
  }
  
  isStateTransitioning = true;
  currentState = STATES.FEED_WATCHING;
  console.log('🔥 STATE: FEED_WATCHING - Monitoring feed for new messages');
  
  try {
    // Navigate to LinkedIn feed and start monitoring
    await navigateToFeed();
    await delay(2000); // Wait for page load
    
       // Start checking for outbox messages (with backoff + jitter; force fast on entry)
    outboxNoChangeStreak = 0;
    scheduleNextOutboxCheck('enter_feed', true);
  } catch (error) {
    console.error('🔥 Error transitioning to feed:', error);
  }
  
  isStateTransitioning = false;
// Now we’re idle; if there’s backlog/realtime buffered, drain it
maybeDrainOutboxQueueIfIdle('feed_watch_ready');
setTimeout(() => maybeDrainOutboxQueueIfIdle('feed_watch_ready_delay'), 400);
}

// Check for messages in outbox while in feed watching state
async function checkForMessages() {
  console.log('🔥 checkForMessages deprecated — using Realtime');
}

// MESSAGE SENDING STATE: Send limited messages then scrape to verify
async function transitionToSendingMessages(messagesFromRealtime = null) {
let hadWork = false;
  // Prevent concurrent message sending or state transitions
  if (isProcessingOutbox || isStateTransitioning) {
    console.log('🔥 Already processing messages or transitioning - skipping');
    return;
  }
  
  isProcessingOutbox = true;
  isStateTransitioning = true;
  
  // Clear any pending message checks from feed watching
  if (checkMessagesTimeout) {
    clearTimeout(checkMessagesTimeout);
    checkMessagesTimeout = null;
  }
  
  currentState = STATES.SENDING_MESSAGES;
  console.log('🔥 STATE: SENDING_MESSAGES - Processing outbox queue');
  
  try {
    // Navigate to messaging page for sending messages
    await navigateToMessaging();
    await delay(5000); // Give more time for page to fully load
    
    // CRITICAL: Stop all scraping during message sending to prevent conflicts
    await chrome.storage.local.set({ 
      outreachos_sync_paused: true,    // Stops messaging_sync.js from scraping
      messageBeingSent: true           // Additional flag for coordination
    });

    // Check for messages in outbox (including retry messages)
 let queue = messagesFromRealtime || [];

if (queue.length === 0) {
  // fallback: query supabase directly for pending rows
  const client = await ensureSupabaseAuth();
  if (client) {
    const { data, error } = await client
      .from('outbox')
      .select('*')
      .in('status', ['pending', 'queued'])
      .order('created_at', { ascending: true })
      .limit(5);
    if (error) {
      console.error('🔥 Failed to fetch pending messages from Supabase:', error.message);
    } else {
      queue = data || [];
    }
  }
}
hadWork = Array.isArray(queue) && queue.length > 0;
if (!hadWork) {
  console.log('🔥 No messages to send - outbox is empty');
  await finishMessageSending(false);
  return;
}

console.log(`🔥 Found ${queue.length} messages in outbox`);
const messagesToSend = queue.slice(0, 2); // Take first 2 messages max
    console.log(`🔥 Sending batch of ${messagesToSend.length} messages`);
    
    for (const msg of messagesToSend) {
      try {
        const attemptText = msg.attempts > 0 ? ` (retry ${msg.attempts})` : '';
        console.log(`🔥 Sending message to ${msg.participant_name}${attemptText}`);
        
        // Ensure we're still on the messaging page
        const tab = await chrome.tabs.get(mainTabId);
        if (!tab.url.includes('/messaging/')) {
          console.log('🔥 Not on messaging page - navigating back');
          await navigateToMessaging();
          await delay(3000);
        }
        
        // Send message via content script
        const res = await sendMessageViaContentScript(msg);
        
      if (res.success) {
  console.log(`✅ Message sent successfully to ${msg.participant_name}`);

  // NEW: Optimistically mark as sent in DB
  try {
    await markOutboxMessage(msg.id, 'sent', null, (msg.attempts || 0) + 1);
  } catch (e) {
    console.warn('Failed to mark sent optimistically:', e?.message || e);
  }

  await delay(8000); // Human-like delay
} else {
          console.log(`❌ Message failed to ${msg.participant_name}:`, res.error);
          // Mark as failed immediately if sending actually failed
          await markOutboxMessage(msg.id, 'failed', res.error, (msg.attempts || 0) + 1);
          
          // If message fails, stop the batch
          console.log('🔥 Stopping batch due to failure');
          break;
        }
      } catch (e) {
        console.error(`❌ Message sending error for ${msg.participant_name}:`, e);
        await markOutboxMessage(msg.id, 'failed', e.message, (msg.attempts || 0) + 1);
        break; // Stop batch on error
      }
    }

 // After sending batch, transition to incremental scraping
console.log('🔥 Batch sent - transitioning to incremental scraping');
await finishMessageSending(true);
    
} catch (error) {
  console.error('🔥 Error in message sending flow:', error);
  await finishMessageSending(hadWork); // Clean up; keep draining only if batch had work
}
}

// Clean up after message sending and start incremental scrape
async function finishMessageSending(hadWork = false) {
  console.log('🔥 Finishing message batch - starting incremental scrape');
  
  // Store that we came from message sending
  await chrome.storage.local.set({ 
    lastStateBeforeScraping: STATES.SENDING_MESSAGES 
  });
  
  // CRITICAL: Resume scraping operations
  await chrome.storage.local.set({ 
    outreachos_sync_paused: false,   // Allow messaging_sync.js to scrape again
    messageBeingSent: false          // Clear coordination flag
  });
  
isProcessingOutbox = false;
// Keep draining across batches only if this batch actually had work
backlogDetected = !!hadWork;

// Wait before transitioning to avoid conflicts
await delay(3000);

// Reset transition flag before deciding next state
isStateTransitioning = false;

if (hadWork) {
  // We sent messages → do incremental scrape to verify
  await transitionToIncrementalScraping();
} else {
  // No messages this batch → go idle, do NOT scrape
  await transitionToFeedWatching();
}
}

// ==================== NAVIGATION FUNCTIONS ====================
// These functions handle navigating the single tab between messaging and feed

// Navigate to LinkedIn messaging page
async function navigateToMessaging() {
  // Ensure tab exists before navigation
  if (!(await ensureMainTabExists())) {
    console.log('🔥 Waiting for tab to be ready for messaging navigation...');
    await delay(3000);
  }
  
  console.log('🔥 Navigating to messaging page...');
  try {
    // Navigate to messaging URL without making tab active (background operation)
    await chrome.tabs.update(mainTabId, { url: LINKEDIN_MESSAGING_URL, active: false });
  } catch (error) {
    console.error('🔥 Error navigating to messaging:', error);
    await ensureMainTabExists(); // Try to recreate tab on error
  }
}

// Navigate to LinkedIn feed page
async function navigateToFeed() {
  // Ensure tab exists before navigation
  if (!(await ensureMainTabExists())) {
    console.log('🔥 Waiting for tab to be ready for feed navigation...');
    await delay(3000);
  }
  
  console.log('🔥 Navigating to feed page...');
  try {
    // Navigate to feed URL without making tab active (background operation)
    await chrome.tabs.update(mainTabId, { url: FEED_URL, active: false });
  } catch (error) {
    console.error('🔥 Error navigating to feed:', error);
    await ensureMainTabExists(); // Try to recreate tab on error
  }
}

// ==================== MESSAGE HANDLERS ====================
// These handle communication between different parts of the extension

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
if (request?.type === "OUTREACHOS_EXTENSION_HANDSHAKE") {
  console.log('🔥 Received handshake from web app');
  chrome.tabs.query({}, (tabs) => {
    // Look for existing pinned LinkedIn messaging tab
    const existingPinned = tabs.find(tab =>
      tab.pinned && tab.url && tab.url.startsWith(LINKEDIN_MESSAGING_URL)
    );

    if (existingPinned) {
      console.log('🔥 Reusing existing pinned LinkedIn tab:', existingPinned.id);
      mainTabId = existingPinned.id;
      chrome.storage.local.set({ outreachos_main_tab_id: existingPinned.id });
    } else {
      // If no pinned tab, look for any LinkedIn messaging tab
      const existingAny = tabs.find(tab =>
        tab.url && tab.url.startsWith(LINKEDIN_MESSAGING_URL)
      );
      if (existingAny) {
        console.log('🔥 Reusing existing LinkedIn tab (not pinned):', existingAny.id);
        mainTabId = existingAny.id;
        chrome.storage.local.set({ outreachos_main_tab_id: existingAny.id });
      } else {
        console.log('🔥 No LinkedIn messaging tab found — creating new one');
        openMainTab();
      }
    }
  });
  sendResponse({ success: true });
  return true;
}

  if (request.action === "OUTREACHOS_SYNC_MESSAGES") {
    chrome.storage.local.get('outreachos_jwt', ({ outreachos_jwt }) => {
      if (!outreachos_jwt) {
        sendResponse({ success: false, error: "No JWT" });
        return;
      }

      // ✅ API OPTIMIZATION: Using environment-aware URL helper
      fetch(getApiUrl('sync-messages'), {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          "Authorization": `Bearer ${outreachos_jwt}`
        },
        body: JSON.stringify(request.payload),
      })
      .then(r => r.json())
      .then(data => sendResponse({ success: true, data }))
      .catch(err => sendResponse({ success: false, error: err.message }));
    });
    return true;
  }

  if (request.action === "OUTREACHOS_GET_TAB_ID") {
    sendResponse({ tabId: sender.tab ? sender.tab.id : null });
    return true;
  }

  if (request.action === "OUTREACHOS_GET_ORDINALS") {
    // ✅ API OPTIMIZATION: Using environment-aware URL helper
    fetch(`${getApiUrl('get-ordinals')}?linkedin_url=${encodeURIComponent(request.linkedin_url)}`)
      .then(r => r.json())
      .then(data => sendResponse({ data }))
      .catch(err => sendResponse({ data: {} }));
    return true;
  }

if (request.action === 'MESSAGING_SCRAPE_DONE') {
  (async () => {
    console.log('🔥 Scrape completed');

    if (currentState === STATES.STARTUP_SCRAPING) {
      // Optional: mark done on success too (harmless if already done)
      try {
        await markFirstSyncDone();
      } catch (e) {}

      isStateTransitioning = false;
      transitionToFeedWatching();
    } else if (currentState === STATES.INCREMENTAL_SCRAPING && !isProcessingOutbox) {
      isStateTransitioning = false;
      transitionToFeedWatching();
    }

    sendResponse({ ok: true });
  })();
  return true;
}

if (request.action === 'BADGE_CHANGED') {
  (async () => {
    const { outreachos_blocked } = await chrome.storage.local.get('outreachos_blocked');
    if (outreachos_blocked) {
      console.log('🔥 Badge change ignored (blocked)');
      sendResponse({ ok: true, blocked: true });
      return;
    }
    console.log('🔥 Badge changed - new messages detected');
    if (currentState === STATES.FEED_WATCHING && !isStateTransitioning) {
      transitionToIncrementalScraping();
    }
    sendResponse({ ok: true });
  })();
  return true;
}
});

// Mid-session sign-out notification (throttled)
chrome.storage.onChanged.addListener((changes, area) => {
  if (area === 'local' && 'outreachos_jwt' in changes) {
    const oldVal = changes.outreachos_jwt.oldValue;
    const newVal = changes.outreachos_jwt.newValue;
    if (oldVal && !newVal) {
      maybeNotifyNoJWT('mid-session-signout');
    }
  }
});

// ==================== TAB EVENT HANDLERS ====================
// These handle tab lifecycle events to maintain our single LinkedIn tab

// Handle tab closure - recreate main tab if it gets closed
chrome.tabs.onRemoved.addListener((tabId) => {
  if (tabId === mainTabId) {
    console.log('🔥 Main LinkedIn tab was closed - will recreate in 30 seconds');
    mainTabId = null;
    
    // Show notification to user
    chrome.notifications.create({
      type: "basic",
      iconUrl: "icons/icon-128.png",
      title: "LinkedIn Tab Closed",
      message: "Your LinkedIn tab was closed. We'll reopen it in 30 seconds to continue syncing.",
    });
    
    // Recreate tab after delay
    setTimeout(openMainTab, 30000);
  }
});

// Handle tab updates - track when user navigates to LinkedIn URLs
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  // Only track tabs that navigate to LinkedIn messaging or feed
  if (tab.url && (tab.url.startsWith(LINKEDIN_MESSAGING_URL) || tab.url.startsWith(FEED_URL))) {
    
    // If we already have a main tab and this is a different tab, ignore it
    if (mainTabId && tabId !== mainTabId) {
      console.log('🔥 Different tab navigated to LinkedIn - ignoring');
      return;
    }
    
    // Update our main tab ID and store it
    mainTabId = tabId;
    chrome.storage.local.set({ outreachos_main_tab_id: tabId });
    console.log('🔥 Updated main tab ID:', tabId);
  }
});

// ==================== BADGE VERIFICATION ====================
// Verify badge detection is working
async function verifyBadgeDetection() {
  // Blocked? Don’t trigger a scrape
  const { outreachos_blocked } = await chrome.storage.local.get('outreachos_blocked');
  if (outreachos_blocked) return;

  if (!mainTabId || currentState !== STATES.FEED_WATCHING) return;

  try {
    chrome.tabs.sendMessage(mainTabId, { action: 'CHECK_BADGE_COUNT' }, (response) => {
      if (chrome.runtime.lastError) {
        console.log('🔥 Badge verification failed:', chrome.runtime.lastError.message);
      } else if (response && response.badgeCount > 0) {
        console.log('🔥 Badge verification found messages:', response.badgeCount);
        if (currentState === STATES.FEED_WATCHING && !isStateTransitioning) {
          console.log('🔥 Missed badge change detected - triggering incremental scrape');
          transitionToIncrementalScraping();
        }
      }
    });
  } catch (error) {
    console.error('🔥 Badge verification error:', error);
  }
}

// Run badge verification every 2 minutes as safety net
setInterval(verifyBadgeDetection, 2 * 60 * 1000);

// ==================== HELPER FUNCTIONS ====================
// Utility functions for API calls and operations

// 🔥 UPDATED: Fetch messages from outbox API (including device-aware retry-eligible failed messages)
// 🔥 Realtime handles outbox now, so no manual fetch
async function fetchOutbox() {
  console.log('🔥 fetchOutbox called — deprecated, use realtime subscription');
  return { data: [] };
}


///RERTY LOGIC FOR OUTBOX
function nextRetryDelayMs(attempts) {
switch (attempts) {
case 1: return 3 * 60 * 1000; // 3 minutes
case 2: return 6 * 60 * 1000; // 6 minutes
case 3: return 15 * 60 * 1000; // 15 minutes
default: return null; // no further retry
}
}



// 🔥 UPDATED: Update message status in database after sending (with retry tracking)
// 🔥 Direct Supabase update instead of proxy API
async function markOutboxMessage(id, status, errorMsg = null, attempts = 0) {
  try {
    const client = await ensureSupabaseAuth();
    if (!client) throw new Error('No Supabase client');

    // Compute retry_at if this is a failure (based on the attempts value you pass in)
    const delayMs = status === 'failed' ? nextRetryDelayMs(attempts) : null;
    const retry_at = delayMs ? new Date(Date.now() + delayMs).toISOString() : null;

    const payload = {
      status,
      error: errorMsg,
      attempts,
      sent_at: status === 'sent' ? new Date().toISOString() : null
    };

    if (status === 'failed') {
      payload.retry_at = retry_at;   // schedule next retry time
    } else if (status === 'sent') {
      payload.retry_at = null;       // clear retry schedule on success
    }

    const { error: updateError } = await client
      .from('outbox')
      .update(payload)
      .eq('id', id);

    if (updateError) {
      console.error('🔥 Failed to update message:', updateError.message);
    } else {
      console.log(`🔥 Updated outbox ${id} → ${status}${retry_at ? ` (retry_at=${retry_at})` : ''}`);
    }
  } catch (err) {
    console.error('🔥 markOutboxMessage error:', err);
  }
}

// Send message via content script (connect.js handles the actual UI interaction)
async function sendMessageViaContentScript(messageObj) {
  return new Promise((resolve) => {
    console.log(`🔥 Sending message ${messageObj.id} to ${messageObj.participant_name} via content script`);
    
    // Check if tab is ready first
    chrome.tabs.get(mainTabId, (tab) => {
      if (chrome.runtime.lastError) {
        console.error('🔥 Tab not found:', chrome.runtime.lastError);
        resolve({ success: false, error: 'Tab not found' });
        return;
      }
      
      console.log('🔥 Tab URL:', tab.url, 'Status:', tab.status);
      
      // Set a timeout
      const timeout = setTimeout(() => {
        console.error(`🔥 Message sending timeout for ${messageObj.participant_name}`);
        resolve({ success: false, error: 'Timeout - no response from content script' });
      }, 20000);
      
      chrome.tabs.sendMessage(mainTabId, {
        action: "OUTREACHOS_OPEN_TYPE_SEND",
        payload: {
          id: messageObj.id,
          name: messageObj.participant_name,
          message: messageObj.message,
          verifyDom: true
        }
      }, (response) => {
        clearTimeout(timeout);
        
        if (chrome.runtime.lastError || !response) {
          const error = chrome.runtime.lastError?.message || 'No response from content script';
          console.error(`🔥 Content script error for ${messageObj.participant_name}:`, error);
          // Mark as failed but let verification system handle it
          resolve({ success: false, error });
        } else {
          console.log(`🔥 Content script response for ${messageObj.participant_name}:`, response);
          resolve(response);
        }
      });
    });
  });
}

// Simple delay utility function
function delay(ms) {
  return new Promise(res => setTimeout(res, ms));
}

////sync update helper
async function markFirstSyncDone() {
  try {
    const { outreachos_jwt } = await chrome.storage.local.get('outreachos_jwt');
    if (!outreachos_jwt) return;

    // ✅ API OPTIMIZATION: Using environment-aware URL helper
    await fetch(getApiUrl('update-sync-progress'), {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${outreachos_jwt}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        first_sync_done: true,
        sync_progress: 50 // mark as complete; prevents null in API updater
      })
    });

    console.log('✅ first_sync_done marked true (fallback)');
  } catch (e) {
    console.log('🔥 markFirstSyncDone error:', e?.message || e);
  }
}
// ==================== 🔥 NEW: WORKFLOW AUTOMATION SYSTEM ====================
// 🔥 NEW: Workflow alarm listener - runs every 60 seconds to check for tasks
chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'workflowAlarm') {
    console.log('🔥 workflowAlarm tick');
    runWorkflowCycle(); // Check for and execute workflow tasks
  }
});

/* ==================== 🔍 NEW: DAILY CONNECTIONS VERIFICATION ==================== */
// Runs once per day before normal workflow tasks.
// Scrapes top 100 "Recently added" connections in a temp tab and updates DB.

async function openTempTab(url) {
  return new Promise((resolve) => {
    chrome.tabs.create({ url, active: false }, async (tab) => {
      try {
        if (tab?.id) {
          await chrome.tabs.update(tab.id, { pinned: true, active: false });
        }
      } catch (e) {}
      resolve(tab);
    });
  });
}

async function scrapeConnectionsInTempTab(topN = 100, attempt = 1) {
  const MAX_ATTEMPTS = 3;
  let tabClosed = false;
  const tab = await openTempTab('https://www.linkedin.com/mynetwork/invite-connect/connections/');
  await delay(5000);

  // Listen for tab close
  function onTabRemoved(closedTabId) {
    if (closedTabId === tab.id) tabClosed = true;
  }
  chrome.tabs.onRemoved.addListener(onTabRemoved);

  let result;
  try {
    // Optional ping in case content script is still loading
    const pingOk = await new Promise((resolve) => {
      try {
        chrome.tabs.sendMessage(tab.id, { action: 'PING' }, (resp) => {
          resolve(!chrome.runtime.lastError && resp?.alive);
        });
      } catch (e) {
        resolve(false);
      }
    });
    if (!pingOk) await delay(1500);

    result = await new Promise((resolve) => {
      let done = false;
      const timeout = setTimeout(() => {
        if (!done) resolve({ success: false, error: 'Timeout SCRAPE_CONNECTIONS' });
      }, 45000);

      try {
        chrome.tabs.sendMessage(tab.id, { action: 'SCRAPE_CONNECTIONS', topN }, (resp) => {
          done = true;
          clearTimeout(timeout);
          resolve(resp || { success: false, error: 'No response from content script' });
        });
      } catch (e) {
        clearTimeout(timeout);
        resolve({ success: false, error: e?.message || 'sendMessage failed' });
      }
    });
  } finally {
    try { chrome.tabs.remove(tab.id); } catch {}
    chrome.tabs.onRemoved.removeListener(onTabRemoved);
  }

  // Retry only if tab was closed before scrape finished
  if (tabClosed && attempt < MAX_ATTEMPTS) {
    console.log(`🔁 Retrying connections scrape (tab closed, attempt ${attempt + 1})...`);
    await delay(2000);
    return scrapeConnectionsInTempTab(topN, attempt + 1);
  }
  return result;
}

async function updateConnectionDatabase(connections) {
  const { outreachos_jwt } = await chrome.storage.local.get('outreachos_jwt');
  if (!outreachos_jwt) throw new Error('No JWT');

  // Chunk uploads defensively
  const CHUNK = 200;
  let summary = { matched: 0, updated: 0, events: 0, aqueue_done: 0, pc_updated: 0 };

  for (let i = 0; i < connections.length; i += CHUNK) {
    const chunk = connections.slice(i, i + CHUNK);
    const resp = await fetch(`${WORKFLOW_API_BASE}/updateConnections`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${outreachos_jwt}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ connections: chunk })
    });

    let data = {};
    try { data = await resp.json(); } catch {}

    if (resp.ok) {
      summary.matched += data?.matched || 0;
      summary.updated += data?.updated || 0;
      summary.events += data?.events || 0;
      summary.aqueue_done += data?.aqueue_done || 0;
      summary.pc_updated += data?.pc_updated || 0;
    } else {
      console.log('🔥 updateConnections API failed:', resp.status, data);
    }
  }

  return summary;
}

async function runConnectionsVerificationIfNeeded() {
  const today = new Date().toISOString().slice(0, 10);
  const { lastConnectionsSyncDate } = await chrome.storage.local.get('lastConnectionsSyncDate');
  if (lastConnectionsSyncDate === today) return false;

  const hasJWT = await checkJWTAndNotify();
  if (!hasJWT) return false;

  console.log('🔍 Running daily connections verification (top 200)...');
  const scrapeRes = await scrapeConnectionsInTempTab(200);
  if (!scrapeRes?.success) {
    console.log('🔥 Connections scrape failed:', scrapeRes?.error);
    return false;
  }

  const { connections = [] } = scrapeRes;
  if (connections.length === 0) {
    console.log('ℹ️ No connections scraped.');
    await chrome.storage.local.set({ lastConnectionsSyncDate: today });
    return true;
  }

  const stats = await updateConnectionDatabase(connections);
  console.log('📊 Connections sync stats:', stats);

  await chrome.storage.local.set({ lastConnectionsSyncDate: today });
  return true;
}
/* ==================== END DAILY CONNECTIONS VERIFICATION ==================== */

// 🔥 NEW: Main workflow cycle - claims tasks from API and executes them
async function runWorkflowCycle() {
  // Blocked? Skip the entire workflow cycle (prevents any Edge/API calls)
  const { outreachos_blocked } = await chrome.storage.local.get('outreachos_blocked');
  if (outreachos_blocked) {
    console.log('🔥 Workflow blocked by min_version — skipping cycle');
    return;
  }

  if (isWorkflowActive) {
    console.log('🔥 Workflow skipped - already running');
    return;
  }

  isWorkflowActive = true;
  console.log('🔥 Workflow cycle starting...');

  try {
    await runConnectionsVerificationIfNeeded();
    await processRetryQueue();

    const task = await claimWorkflowTask();
    if (task) {
      console.log('🔥 Claimed workflow task:', task.action_type, 'for', task.linkedin_url);
      await executeWorkflowTask(task);
    } else {
      console.log('🔥 No workflow tasks available');
    }
  } catch (error) {
    console.error('🔥 Workflow cycle error:', error);
  } finally {
    isWorkflowActive = false;
  }
}
// 🔥 NEW: Claim task from workflow API
async function claimWorkflowTask() {
  const { outreachos_jwt } = await chrome.storage.local.get('outreachos_jwt');
  if (!outreachos_jwt) {
    console.log('🔥 No JWT for workflow - skipping');
    return null;
  }

  const now = Date.now();
  if (now < nextWorkflowClaimAt) {
    const inMs = nextWorkflowClaimAt - now;
    console.log(`⏱️ Skipping claim-task; next allowed in ${Math.round(inMs / 1000)}s`);
    return null;
  }

  try {
    const response = await fetch(`${WORKFLOW_API_BASE}/claim-task`, {
      headers: { 'Authorization': `Bearer ${outreachos_jwt}` }
    });

    if (!response.ok) {
      console.log('🔥 Workflow claim failed:', response.status);
      // On transient errors, try again after a jittered short delay (2 minutes)
      nextWorkflowClaimAt = now + jitterMs(2 * 60 * 1000, WORKFLOW_BACKOFF.JITTER_PCT);
      return null;
    }

    const data = await response.json();
    const task = data.task || null;

    if (task) {
      // Task found → drain quickly for a short burst
      workflowNoTaskStreak = 0;
      workflowBackoffIndex = -1;
      nextWorkflowClaimAt = now + jitterMs(WORKFLOW_BACKOFF.ACTIVE_FAST_MS, WORKFLOW_BACKOFF.JITTER_PCT); // ~1 minute
      return task;
    }

    // No task found → apply staged backoff:
    // Base cadence: every 3–6 minutes with jitter
    // After NO_TASK_THRESHOLD consecutive "no task", escalate to 15m → then 60m
    if (workflowBackoffIndex < 0) {
      workflowNoTaskStreak += 1;
      if (workflowNoTaskStreak >= WORKFLOW_BACKOFF.NO_TASK_THRESHOLD) {
        workflowBackoffIndex = 0; // jump to 15m tier
        workflowNoTaskStreak = 0;
      }
    } else if (workflowBackoffIndex < WORKFLOW_BACKOFF.STEPS_MS.length - 1) {
      workflowBackoffIndex += 1; // escalate to next tier (60m)
    }

    let delayMs;
    if (workflowBackoffIndex < 0) {
      const base = WORKFLOW_BACKOFF.BASE_MIN_MS + Math.random() * (WORKFLOW_BACKOFF.BASE_MAX_MS - WORKFLOW_BACKOFF.BASE_MIN_MS);
      delayMs = jitterMs(base, WORKFLOW_BACKOFF.JITTER_PCT); // ~3–6m
    } else {
      delayMs = jitterMs(WORKFLOW_BACKOFF.STEPS_MS[workflowBackoffIndex], WORKFLOW_BACKOFF.JITTER_PCT); // 15m or 60m
    }

    nextWorkflowClaimAt = now + Math.min(delayMs, WORKFLOW_BACKOFF.MAX_MS);
    const mins = Math.round((nextWorkflowClaimAt - now) / 60000);
    console.log(`🕒 No workflow tasks. Backoff tier=${workflowBackoffIndex}, next claim in ~${mins}m`);
    return null;
  } catch (error) {
    console.error('🔥 Task claim error:', error);
    // Network error → short backoff (2 minutes) to avoid tight loops
    nextWorkflowClaimAt = now + jitterMs(2 * 60 * 1000, WORKFLOW_BACKOFF.JITTER_PCT);
    return null;
  }
}

// 🔥 FIXED: Execute workflow task - 

// === REPLACE THIS ENTIRE FUNCTION IN background.js ===
async function temporaryFocus(tabId) {
  try {
    const [prev] = await chrome.tabs.query({ active: true, lastFocusedWindow: true });
    const prevFocus = prev ? { tabId: prev.id, windowId: prev.windowId } : null;

    const tab = await chrome.tabs.get(tabId);
    await chrome.windows.update(tab.windowId, { focused: true });
    await chrome.tabs.update(tabId, { active: true });

    return prevFocus;
  } catch (e) {
    console.log('🔥 Focus error:', e?.message || e);
    return null;
  }
}

async function restoreFocus(prevFocus) {
  if (!prevFocus || !prevFocus.tabId || !prevFocus.windowId) return;
  try {
    await chrome.windows.update(prevFocus.windowId, { focused: true });
    await chrome.tabs.update(prevFocus.tabId, { active: true });
  } catch (e) {
    console.log('🔥 Restore focus error:', e?.message || e);
  }
}

let isListeningForNewTaskTab = false;
let newlyCreatedTabId = null;

chrome.tabs.onCreated.addListener((tab) => {
  if (isListeningForNewTaskTab && tab.url && tab.url.includes('linkedin.com')) {
    console.log(`[WORKFLOW] Detected new LinkedIn tab created during task: ${tab.id}`);
    newlyCreatedTabId = tab.id;
    isListeningForNewTaskTab = false; // Stop listening after finding one
  }
});

async function executeWorkflowTask(task) {
  console.log('🔥 Executing workflow task:', task.action_type, 'for', task.linkedin_url);
  let bgTabId = null;
  newlyCreatedTabId = null; // Reset for this task

  try {
    const targetUrl = task.conversation_url || task.linkedin_url;
    const isDirectToThread = !!task.conversation_url;

    console.log(`[WORKFLOW] DM: Navigating to ${isDirectToThread ? 'thread' : 'profile'}: ${targetUrl}`);
    const tab = await chrome.tabs.create({ url: targetUrl, active: false, pinned: true });
    bgTabId = tab.id;
    await chrome.tabs.move(bgTabId, { index: 1 });
    const ready = await ensureConnectScriptReady(bgTabId, 20000);
    if (!ready) {
      await completeWorkflowTask(task, false, {}, 'content script not ready');
      return;
    }

    if (task.action_type.toUpperCase() === 'DM') {
      if (isDirectToThread) {
        const result = await new Promise(res => {
          const t = setTimeout(() => res({ success: false, error: 'timeout: SEND_ON_THREAD_PAGE' }), 60000);
          chrome.tabs.sendMessage(bgTabId, { type: 'SEND_ON_THREAD_PAGE', payload: task }, r => {
            clearTimeout(t);
            res(r || { success: false, error: 'no response from SEND_ON_THREAD_PAGE' });
          });
        });
        await completeWorkflowTask(task, !!result.success, result.evidence, result.error);
      } else {
        const attemptResult = await new Promise(res => {
          const t = setTimeout(() => res({ success: false, error: 'timeout: ATTEMPT_DM_BACKGROUND' }), 30000);
          chrome.tabs.sendMessage(bgTabId, { type: 'ATTEMPT_DM_BACKGROUND', payload: task }, r => {
            clearTimeout(t);
            res(r || { success: false, error: 'no response from ATTEMPT_DM_BACKGROUND' });
          });
        });

        if (attemptResult.success) {
          await completeWorkflowTask(task, true, attemptResult.evidence, null);
        } else {
          isListeningForNewTaskTab = true; // Start listening before the foreground action
          const previousFocus = await temporaryFocus(bgTabId);
          const execResult = await new Promise(res => {
            const t = setTimeout(() => res({ success: false, error: 'timeout: EXECUTE_DM_FOREGROUND' }), 60000);
            chrome.tabs.sendMessage(bgTabId, { type: 'EXECUTE_DM_FOREGROUND', payload: task }, r => {
              clearTimeout(t);
              res(r || { success: false, error: 'no response from EXECUTE_DM_FOREGROUND' });
            });
          });
          await restoreFocus(previousFocus);
          isListeningForNewTaskTab = false; // Stop listening after the action
          await completeWorkflowTask(task, !!execResult.success, execResult.evidence, execResult.error);
        }
      }
    } else {
      const result = await new Promise(res => {
        const t = setTimeout(() => res({ success: false, error: `timeout: ${task.action_type}` }), 60000);
        chrome.tabs.sendMessage(bgTabId, { type: `OUTREACHOS_WORKFLOW_${task.action_type.toUpperCase()}`, payload: task }, r => {
          clearTimeout(t);
          res(r || { success: false, error: 'no response from workflow action' });
        });
      });
      await completeWorkflowTask(task, !!result.success, result.evidence, result.error);
    }
  } catch (err) {
    console.error('🔥 Task execution error:', err);
    await addTaskToRetryQueue(task);
  } finally {
    if (bgTabId) {
      await chrome.tabs.remove(bgTabId).catch(() => {});
    }
    if (newlyCreatedTabId) {
      await chrome.tabs.remove(newlyCreatedTabId).catch(() => {});
      newlyCreatedTabId = null;
    }
    isListeningForNewTaskTab = false; // Ensure listener is always turned off
  }
}


// ADD these helpers anywhere below in background.js

// ===================================================================================================
// 🔥 FIX: Enhanced content script readiness detection with better injection and timing
// This fixes the "Could not establish connection. Receiving end does not exist" error
// ===================================================================================================
async function ensureConnectScriptReady(tabId, timeoutMs = 15000) {
  const start = Date.now();
  console.log(`🔥 Ensuring content script ready for tab ${tabId}...`);

  // 1) Wait for tab complete with better error handling
  const tabReady = await waitForTabComplete(tabId, Math.min(timeoutMs, 8000));
  if (!tabReady) {
    console.log('🔥 Tab did not reach complete in time - checking if still exists');
    try {
      const tab = await chrome.tabs.get(tabId);
      console.log('🔥 Tab exists but not complete - status:', tab.status, 'url:', tab.url);
    } catch (e) {
      console.log('🔥 Tab no longer exists:', e?.message || e);
      return false;
    }
  }

  // 2) Try initial ping with longer timeout
  console.log('🔥 Pinging content script...');
  const pingOK = await pingConnect(tabId, 5000); // Increased from 3000ms
  if (pingOK) {
    console.log('🔥 Content script responding to ping - ready!');
    return true;
  }

  console.log('🔥 Content script not responding - attempting injection...');

  // 3) Enhanced fallback: inject both connect.js and messaging-sync.js if needed
  try {
    // First try to inject the main content script
    await chrome.scripting.executeScript({
      target: { tabId },
      files: ['connect.js']
    });
    console.log('🔥 Injected connect.js via scripting API');
    
    // Also inject messaging-sync.js if we're on messaging page
    const tab = await chrome.tabs.get(tabId);
    if (tab.url && tab.url.includes('/messaging/')) {
      try {
        await chrome.scripting.executeScript({
          target: { tabId },
          files: ['messaging-sync.js']
        });
        console.log('🔥 Also injected messaging-sync.js for messaging page');
      } catch (e) {
        console.log('🔥 messaging-sync.js injection failed (non-critical):', e?.message || e);
      }
    }
    
    // Wait a bit for scripts to initialize
    await delay(2000);
    
  } catch (e) {
    console.log('🔥 Script injection failed:', e?.message || e);
    return false;
  }

  // 4) Final ping after injection with remaining time
  const remainingTime = Math.max(2000, timeoutMs - (Date.now() - start));
  console.log(`🔥 Final ping attempt with ${remainingTime}ms timeout...`);
  const pingAfterInject = await pingConnect(tabId, remainingTime);
  
  if (pingAfterInject) {
    console.log('🔥 Content script ready after injection!');
  } else {
    console.log('🔥 Content script still not responding after injection');
  }
  
  return pingAfterInject;
}

// ===================================================================================================
// 🔥 FIX: Enhanced ping function with better error logging and timing
// This helps diagnose content script connection issues
// ===================================================================================================
function pingConnect(tabId, waitMs = 3000) {
  return new Promise((resolve) => {
    let done = false;
    let attempts = 0;
    const end = Date.now() + waitMs;

    const tryPing = () => {
      attempts++;
      if (Date.now() > end) {
        if (!done) {
          console.log(`🔥 Ping timeout after ${attempts} attempts for tab ${tabId}`);
          resolve(false);
        }
        return;
      }
      
      try {
        chrome.tabs.sendMessage(tabId, { action: 'PING' }, (resp) => {
          if (chrome.runtime.lastError) {
            console.log(`🔥 Ping attempt ${attempts} error:`, chrome.runtime.lastError.message);
            setTimeout(tryPing, 300);
          } else if (resp?.alive) {
            console.log(`🔥 Ping successful after ${attempts} attempts`);
            done = true;
            resolve(true);
          } else {
            console.log(`🔥 Ping attempt ${attempts} - no response or not alive`);
            setTimeout(tryPing, 300);
          }
        });
      } catch (e) {
        console.log(`🔥 Ping attempt ${attempts} exception:`, e?.message || e);
        setTimeout(tryPing, 300);
      }
    };

    tryPing();
  });
}

function waitForTabComplete(tabId, timeoutMs = 8000) {
  return new Promise((resolve) => {
    let settled = false;
    const check = async () => {
      try {
        const tab = await chrome.tabs.get(tabId);
        if (tab?.status === 'complete') {
          if (!settled) { settled = true; resolve(true); }
          return;
        }
      } catch {}
      if (!settled) setTimeout(check, 250);
    };

    const to = setTimeout(() => {
      if (!settled) { settled = true; resolve(false); }
    }, timeoutMs);

    check();
  });
}
// 🔥 NEW: Complete task via API (reports success/failure back to database)
async function completeWorkflowTask(task, success, evidence = {}, error = null) {
  const { outreachos_jwt } = await chrome.storage.local.get('outreachos_jwt');
  if (!outreachos_jwt) {
    console.log('🔥 No JWT for task completion');
    return;
  }
  
  try {
    const response = await fetch(`${WORKFLOW_API_BASE}/complete-task`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${outreachos_jwt}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        queue: task.queue, // 'action' or 'message'
        id: task.id,
        success,
        evidence,
        error
      })
    });
    
    if (response.ok) {
  console.log('🔥 Task completed:', task.action_type, success ? 'SUCCESS' : 'FAILED', error || '');

    } else {
      console.log('🔥 Task completion API failed:', response.status);
      // If completion fails and task failed, add to retry queue
      if (!success) await addTaskToRetryQueue(task);
    }
  } catch (err) {
    console.error('🔥 Task completion error:', err);
    // If completion fails and task failed, add to retry queue
    if (!success) await addTaskToRetryQueue(task);
  }
}

// 🔥 NEW: Retry queue management - handles failed tasks with exponential backoff
async function addTaskToRetryQueue(task) {
  const retryTask = {
    ...task,
    retryCount: (task.retryCount || 0) + 1,
    nextRetryTime: Date.now() + (Math.pow(2, (task.retryCount || 0)) * 60000) // 1min, 2min, 4min, 8min...
  };
  
  // Only retry up to 3 times
  if (retryTask.retryCount <= 3) {
    workflowQueue.push(retryTask);
    await saveWorkflowQueue();
    console.log('🔥 Added task to retry queue:', retryTask.action_type, 'retry', retryTask.retryCount);
  } else {
    console.log('🔥 Task exceeded max retries:', task.action_type);
  }
}

// 🔥 NEW: Process retry queue - executes failed tasks when retry time is reached
async function processRetryQueue() {
  const now = Date.now();
  const readyTasks = workflowQueue.filter(task => task.nextRetryTime <= now);
  
  if (readyTasks.length > 0) {
    console.log('🔥 Processing retry queue:', readyTasks.length, 'tasks ready');
  }
  
  // Process one retry task at a time to avoid overwhelming
  for (const task of readyTasks.slice(0, 1)) {
    console.log('🔥 Retrying task:', task.action_type, 'attempt', task.retryCount);
    workflowQueue = workflowQueue.filter(t => t.id !== task.id);
    await executeWorkflowTask(task);
  }
  
  // Save updated queue
  if (readyTasks.length > 0) {
    await saveWorkflowQueue();
  }
}

// 🔥 NEW: Persistence functions - save/load workflow queue to survive browser restarts
async function saveWorkflowQueue() {
  await chrome.storage.local.set({ workflowQueue });
  console.log('🔥 Workflow queue saved:', workflowQueue.length, 'tasks');
}

async function loadWorkflowQueue() {
  const { workflowQueue: saved } = await chrome.storage.local.get('workflowQueue');
  workflowQueue = saved || [];
  console.log('🔥 Workflow queue loaded:', workflowQueue.length, 'tasks');
}

// 🔥 NEW: External message handler for web app token setting
chrome.runtime.onMessageExternal.addListener((request, sender, sendResponse) => {
  if (request.type === 'SET_TOKEN') {
    chrome.storage.local.set({ outreachos_jwt: request.token }, () => {
      console.log('🔥 JWT token set from web app');
      sendResponse({ success: true });
    });
    return true;
  }
});

// ==================== HEALTH CHECK & STARTUP ====================
// Periodic maintenance and initialization

// Health check: Reload LinkedIn tab every 5 minutes to prevent issues
setInterval(() => {
  if (mainTabId) {
    chrome.tabs.reload(mainTabId, {}, () => {
      console.log("🔥 LinkedIn tab reloaded for health check");
    });
  }
}, 5 * 60 * 1000); // 5 minutes

// Start the extension flow after a short delay
setTimeout(() => {
  console.log('🔥 Starting extension flow...');
  startupFlow();
}, 2000);

// Init Supabase Outbox + version check after startup
setTimeout(() => {
  console.log('🔥 Setting up Outbox Realtime subscription…');
  setupOutboxSubscription();

  // Also start version-check subscription
  setupVersionSubscription();

  // Do a one-time fetch at boot to catch outdated versions immediately
  initialMinVersionCheck();
}, 3000);
// Kick Lane 2 once on boot so you see logs immediately
setTimeout(() => {
  try {
    console.log('🔥 Kickoff: running workflow once');
    runWorkflowCycle();
  } catch (e) {
    console.log('Kickoff error:', e?.message || e);
  }
}, 8000);

// ==================== REALTIME OUTBOX ====================
// Helper to get the extension's user id (from JWT)
async function getExtensionUserId() {
  const { outreachos_jwt } = await chrome.storage.local.get('outreachos_jwt');
  const { data: { user } } = await supabase.auth.getUser(outreachos_jwt);
  return user?.id;
}

async function setupOutboxSubscription() {
  const client = await ensureSupabaseAuth();
  if (!client) return;

  // Ensure Realtime socket uses the current JWT and sanity-check RLS visibility
  try {
    client.realtime.connect();

    const { data: sanityRows, error: sanityErr } = await client
      .from('outbox')
      .select('id,status,created_at')
      .in('status', ['pending', 'queued'])
      .order('created_at', { ascending: false })
      .limit(1);

    console.log('DEBUG sanity SELECT (pending/queued)=', sanityRows?.length ?? 0, sanityErr?.message || '');
    if ((sanityRows?.length || 0) > 0) {
      console.log('DEBUG bootstrap: pending/queued rows detected — will send when idle');
      backlogDetected = true;
      maybeDrainOutboxQueueIfIdle('sanity_bootstrap');
    }
  } catch (e) {
    console.log('DEBUG sanity SELECT error:', e?.message || e);
  }

  // Main subscription for outbox table (RLS protected)
  client.channel('outbox-changes')
    .on(
      'postgres_changes',
      { event: '*', schema: 'public', table: 'outbox' },
      (payload) => {
        console.log('DEBUG: Raw realtime payload:', payload);
        const row = payload.new;
        if (!row) return;

        // Normal insert (new pending message)
        if (payload.eventType === 'INSERT' && row.status === 'pending') {
          console.log('🔥 New pending message for user (buffered):', row);
          outboxEventBuffer.push(row);
          backlogDetected = true;
          maybeDrainOutboxQueueIfIdle('rt_insert_pending');
        }
        if (payload.eventType === 'UPDATE') {
          // Explicit re-queue for retry
          if (row.status === 'queued') {
            console.log('🔥 Re-queued message for retry (buffered):', row);
            outboxEventBuffer.push(row);
            backlogDetected = true;
            maybeDrainOutboxQueueIfIdle('rt_update_queued');
          }
          // Timed retry using retry_at
          if (row.status === 'failed' && row.retry_at && new Date(row.retry_at) <= new Date()) {
            console.log('🔥 Retry-at hit (buffered):', row);
            outboxEventBuffer.push(row);
            backlogDetected = true;
            maybeDrainOutboxQueueIfIdle('rt_update_retryat');
          }
        }
      }
    )
    .subscribe((status) => {
      console.log('🔥 Outbox channel status:', status);
      if (status === 'SUBSCRIBED') {
        setTimeout(() => {
          console.log('DEBUG bootstrap on SUBSCRIBED: try drain once (200ms)');
          maybeDrainOutboxQueueIfIdle('subscribed_bootstrap_200ms');
        }, 200);
        setTimeout(() => {
          console.log('DEBUG bootstrap on SUBSCRIBED: try drain once (800ms)');
          maybeDrainOutboxQueueIfIdle('subscribed_bootstrap_800ms');
        }, 800);
      }
    });

  // ADD: Outbox events subscription (no RLS issues, always fires)
getExtensionUserId().then((extensionUserId) => {
  if (!extensionUserId) {
    console.warn('No extension user id for outbox_events subscription');
    return;
  }
  client.channel('outbox-events')
    .on(
      'postgres_changes',
      { event: 'INSERT', schema: 'public', table: 'outbox_events' },
      (payload) => {
        const event = payload.new;
        // Always log what we get
        console.log('DEBUG: Outbox event payload:', event, 'extensionUserId:', extensionUserId);
        if (event.user_id === extensionUserId) {
          console.log('DEBUG: Outbox event detected for this user:', event);
          backlogDetected = true;
          maybeDrainOutboxQueueIfIdle('outbox_event_insert');
        } else {
          console.log('DEBUG: Outbox event for different user:', event.user_id, extensionUserId);
        }
      }
    )
    .subscribe((status) => {
      console.log('🔥 Outbox events channel status:', status);
    });
});

  console.log('🔥 Outbox realtime subscription established');
}

console.log("🔥 OutreachOS background.js fully loaded and ready");

// ==================== PROCESS FLOW SUMMARY ====================
/*
🔥 COMPLETE EXTENSION FLOW EXPLANATION:

1. MESSAGING SYSTEM (Original - Preserved):
   - Uses existing state machine: STARTUP_SCRAPING → INCREMENTAL_SCRAPING → FEED_WATCHING → SENDING_MESSAGES
   - Sends messages via "OUTREACHOS_OPEN_TYPE_SEND" action to connect.js
   - Uses main pinned LinkedIn tab for all messaging operations
   - Handles outbox messages, retry logic, and verification

2. WORKFLOW SYSTEM (New - Parallel):
   - Runs independently every 60 seconds via chrome.alarms
   - Claims tasks from /api/workflow/claim-task
   - Executes via "OUTREACHOS_WORKFLOW_VIEW/FOLLOW/CONNECT/DM" actions to connect.js
   - Uses temporary PINNED tabs that are ALWAYS closed after task completion
   - Completes tasks via /api/workflow/complete-task
   - Has its own retry queue with exponential backoff

   🔍 Daily connections verification (Added):
   - Once per day, before normal tasks, scrapes top 100 "Recently added" connections
   - Sends to /api/workflow/updateConnections to mark existing prospects as connected

3. KEY DIFFERENCES:
   - Messaging: Uses main pinned tab, state machine coordination, existing message queue
   - Workflow: Uses temporary PINNED tabs, independent timing, separate API endpoints
   - Both systems coordinate to avoid conflicts during message sending

4. TAB MANAGEMENT (Fixes applied):
   ✅ On boot/enable: Reuse any existing pinned linkedin.com tab
   ✅ Create a new pinned tab only if none exists

5. NOTIFICATION UX:
   ✅ "No JWT" notification is throttled (5 min cooldown)
   ✅ If token is invalid/expired on startup, show notification and do NOT trigger full scrape
*/